
// modules

use crate::prelude::*;

use crate::game::{attack, pawn};

// types

pub struct Pawn_Info {

   pub inc_sd: i8,

   pub pawn_sd: BB,
   pub pawn_xd: BB,

   pub atk_sd: BB,
   pub atk_xd: BB,

   lever_sd: BB,
   wall_xd:  BB,

   pub blocked_sd: BB,

   weak_sd: BB,
}

// functions

impl Pawn_Info {

   pub fn new(bd: &Board, sd: Side) -> Self {

      let xd = sd.opp();

      let inc_sd = pawn::inc_move(sd);

      let pawn_sd = bd.pawn(sd);
      let pawn_xd = bd.pawn(xd);
      let pawns   = pawn_sd | pawn_xd;

      let atk_sd = BB::from(pawn::caps_froms(bd, sd));
      let atk_xd = BB::from(pawn::caps_froms(bd, xd));

      let lever_sd = pawn_sd & atk_xd;
      let wall_xd  = pawn_xd & atk_xd;

      let stop       = pawns | (atk_xd - atk_sd);
      let blocked_sd = (pawn_sd & stop   .shift(-inc_sd)) - atk_xd;

      // weak squares

      let allowed = BB::pawns() - stop;

      let mut pawn = pawn_sd;
      let mut weak = BB::full();

      while !pawn.is_empty() {
         weak -= pawn::attacks(pawn, sd);
         pawn = allowed & pawn::moves(pawn, sd);
      }

      let weak_sd = blocked_sd & weak;

      Self {

         inc_sd,

         pawn_sd,
         pawn_xd,

         atk_sd,
         atk_xd,

         lever_sd,
         wall_xd,

         blocked_sd,

         weak_sd,
      }
   }

   pub fn is_chain(&self, sq: Square, bd: &Board) -> bool {
      self.atk_sd.has(sq)
   }

   pub fn is_weak(&self, sq: Square, bd: &Board) -> bool {
      self.weak_sd.has(sq)
   }

   pub fn is_open(&self, sq: Square, bd: &Board) -> bool {
      let pawns = (self.pawn_sd - self.lever_sd) | self.wall_xd;
      (pawns & bd.global.table_bb.file(sq)).is_empty()
   }
}

pub fn is_isolated(sq: Square, sd: Side, bd: &Board) -> bool {
   (bd.pawn(sd) & bd.global.table_bb.files(sq)).is_empty()
}

pub fn is_doubled(sq: Square, sd: Side, bd: &Board) -> bool {
   let table_bb = &bd.global.table_bb;
   !(bd.pawn(sd) & table_bb.file(sq) & table_bb.front(sq, sd)).is_empty()
}

pub fn is_duo(sq: Square, sd: Side, bd: &Board) -> bool {
   let table_bb = &bd.global.table_bb;
   !(bd.pawn(sd) & table_bb.files(sq) & table_bb.rank(sq)).is_empty()
}

pub fn is_passed(sq: Square, sd: Side, bd: &Board) -> bool {

   let table_bb = &bd.global.table_bb;

   let files = table_bb.file(sq) | table_bb.files(sq);

   (bd.pawn(sd.opp()) & files & table_bb.front(sq, sd)).is_empty() &&
   !is_doubled(sq, sd, bd)
}

pub fn passed_is_free(sq: Square, sd: Side, bd: &Board) -> bool {

   let table_bb = &bd.global.table_bb;

   let path = table_bb.file(sq) & table_bb.front(sq, sd);

   BB::is_disjoint(path, bd.side(sd.opp())) &&
   !attack::is_attacked_tos(bd, path, sd.opp(), bd.all() - BB::square(sq))
}

