
// modules

use crate::prelude::*;

use super::{eval, order, see, Depth, Eval, Eval_Cache, Indexer, Move_Long, Score, Time_Control, TT};

use order::{Index_PT, Killer, Stat, Table};

use crate::game::{self, gen_, legal, Key, Legal_Info, Undo};

use crate::util::prelude::*;
use crate::util::{self, data::Stack_2, random, thread::AtomicF32};

use std::{ops, time};
use std::sync::{atomic, Mutex};
use atomic::Ordering::*;

// constants

const Depth_Max: Ply = Ply(99);
const Ply_Max:   Ply = Ply(Ply_Size - 2);
const Ply_Size:  u8  = 64;

// types

pub struct Input {

   pub thread: ID,

   pub depth: Option<Ply>,
   pub ply:   Option<Ply>,
   pub node:  Option<Node>,
   pub time:  Option<Time>,
   pub tc:    Option<Time_Control>,

   pub score: bool,
}

#[derive(Clone, Default)]
pub struct Output {
   pub best: Best,
   pub curr: Current,
}

#[derive(Clone)]
pub struct Best {
   pub move_: Option<Move>,
   pub score: Option<(Score, Bound)>,
   pub depth: Ply,
   pub pv:    Vec<Move>,
}

#[derive(Clone)]
pub struct Current {
   pub depth:   Ply,
   pub ply_max: Ply,
   pub node:    Node,
   pub time:    Time,
   pub speed:   Option<Time>,
}

#[repr(align(128))]
pub struct SG<'e> {

   input: Input,

   board: Board,
   start: time::Instant,

   limit_depth: Ply,
   limit_ply:   Ply,
   limit_node:  Option<Node>,
   limit_time:  Option<Time>,
   alloc_tc:    Option<Alloc>,

   pv_move:  Option<Move>,
   tt_depth: Depth, // not used

   tt:     &'e TT,
   shared: Shared,
   main:   Mutex<Main>,
}

#[repr(align(128))]
pub struct Shared {

   ply_max: atomic::AtomicU8,
   node:    atomic::AtomicU64,

   factor: AtomicF32,
   abort:  atomic::AtomicBool,
   depth:  atomic::AtomicU8,
}

pub struct Main {

   output:    Output,
   last_best: Best,

   depth:  Ply,
   factor: f32,
}

#[repr(align(128))]
struct SL<'s> {

   id: ID,
   sg: &'s SG<'s>,

   rand: random::Random_Xor,

   best: Best,
   curr: Current,

   depth:   Ply,
   ply:     Ply,
   ply_max: Ply,
   period:  Node,

   board: Board,
   list:  Vec<Move>,

   todo: Stack_2<Move, 8_192>,
   score:Stack_2<u16,  8_192>,
   done: Stack_2<Move_Long, 1_024>,
   dup:  Stack_2<Move, 1_024>,
   bad:  Stack_2<Move, 1_024>,

   index:  [u16;               Ply_Size as usize],
   move_:  [Option<Move_Long>; Ply_Size as usize],
   eval:   [Option<Score>;     Ply_Size as usize],
   pv:     [Vec<Move>;         Ply_Size as usize],
   killer: [Killer;            Ply_Size as usize],

   hist_0: [[Table<Index_PT, Stat>; 2]; Side::Size as usize],
   hist_1: [Table<Index_PT, Table<Index_PT, Stat>>; Side::Size as usize],
   hist_2: [Table<Index_PT, Table<Index_PT, Stat>>; Side::Size as usize],

   eval_cache: Eval_Cache,
}

#[derive(Clone, Copy)]
struct Alloc {
   time: Time,
   max:  Time,
}

struct Local {

   ab:      AB,
   depth:   Depth,
   pv_node: bool,

   move_: Option<Move_Long>,
   score: Score,
   skip:  bool,

   tt_key:   Option<Key>,
   pv_move:  Option<Move>,
   tt_depth: Depth,
   tt_score: (Score, Score),

   li:   Option<Legal_Info>,
   eval: Option<Score>,

   fp_node:  bool,
   fp_score: Option<Score>,

   stage: Stage,
   stop:  bool,
}

pub struct Table_Search {
   lmr: [[u8; Self::Late_Size as usize]; Self::Depth_Size as usize],
}

struct Frame {

   undo: Undo,

   fp_todo: u16,
   fp_score:u16,
   fp_done: u16,
   fp_dup:  u16,
   fp_bad:  u16,
}

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
enum Stage {
   All,
   PV,
   Capture,
   Promotion,
   Killer,
   Quiet,
   Bad,
}

#[derive(Clone, Copy)]
struct AB {
   pub alpha: Score,
   pub beta:  Score,
}

struct Move_Info {

   move_: Move,
   stage: Stage,

   is_cap:   bool,
   is_prom:  bool,
   is_check: Option<bool>,
   is_recap: Option<bool>,
   is_safe:  Option<bool>,
}

#[derive(Clone, Copy, PartialEq, Eq)]
pub enum Bound { Upper, Lower, Exact }

type Result<T> = std::result::Result<T, Abort>;

#[must_use]
#[derive(Clone, Copy, PartialEq, Eq)]
struct Abort;

#[must_use]
#[derive(Clone, Copy, PartialEq, Eq)]
struct Return;

type ID = u8;

// functions

impl<'e> SG<'e> {

   pub fn new(board: Board, input: Input, tt: &'e TT) -> Self {

      debug_assert!(board.can_play());

      Self {

         input,

         board,
         start: time::Instant::now(),

         limit_depth: Depth_Max,
         limit_ply:   Ply_Max,
         limit_node:  None,
         limit_time:  None,
         alloc_tc:    None,

         pv_move:  None,
         tt_depth: Depth(0),

         tt,
         shared: Shared::new(),
         main:   Main::new().into(),
      }
   }

#[must_use]
   pub fn run(&mut self) -> Output {

      // legal root moves

      let list = legal::legals(&self.board);

      if list.size() == 1 && !self.input.score {
         let mut so = Output::default();
         so.best.move_ = Some(list[0]);
         return so;
      }

      // time allocation

      let lag = self.board.global.time_lag.load();

      if let Some(depth) = self.input.depth {
         self.limit_depth = depth.min(Depth_Max);
      }

      if let Some(ply) = self.input.ply {
         self.limit_ply = ply.min(Ply_Max);
      }

      if let Some(node) = self.input.node {
         self.limit_node = Some(node);
      }

      if let Some(time) = self.input.time {
         self.limit_time = Some((time - lag).max(0.0));
      }

      if let Some(Time_Control { moves, time, mut inc }) = self.input.tc {

         inc -= lag;

         let moves  = (moves.unwrap_or(30) as f32).min(30.0);
         let factor = 1.3;
         let alloc  = (time / moves + inc) * factor;
         let max    = alloc * 2.0;

         self.alloc_tc = Some(Alloc {
            time: alloc.min(time + inc).max(0.0),
            max:  max  .min(time + inc).max(0.0),
         });
      }

      // transposition table

      if let Some(to) = self.tt.probe(self.board.key()) {
         self.pv_move  = to.move_;
         self.tt_depth = Depth::from_int(to.d_min);
      }

      {
         let mut main = self.main.lock().unwrap();
         main.output.best.move_ = self.pv_move;
      }

      // transfert control to thread

      std::thread::scope(|scope| {

         let sg = &self;

         // spawn

         for id in 0 .. sg.input.thread {

            scope.spawn(move || {
               let sl = SL::new(id, sg);
               let _ = sl.run();
            });
         }

      });

      let mut so = {
         let main = self.main.lock().unwrap();
         main.output.clone()
      };

      self.update(&mut so.curr, Ply(0), 0);
      self.disp_time(&so.curr);

      so
   }

   fn update(&self, curr: &mut Current, ply_max: Ply, node: Node) {

      curr.ply_max = self.ply_max(ply_max);
      curr.node    = self.node(node);
      curr.time    = self.time();
      curr.speed   = (curr.time >= 1E-3).then(|| curr.node as Time / curr.time);
   }

   fn disp_move(&self, best: &Best, curr: &Current) {

      debug_assert!(best.depth == curr.depth);

      let mut line = "info".to_owned();

      line += &format!(" depth {}", best.depth);
      if curr.ply_max != Ply(0) { line += &format!(" seldepth {}", curr.ply_max) }

      if let Some((score, bound)) = best.score {

         line += &if score.is_win() {
            format!(" score mate +{}", (score.ply().val() + 1) / 2)
         } else if score.is_loss() {
            format!(" score mate -{}", (score.ply().val() + 1) / 2)
         } else {
            format!(" score cp {}", score.val())
         };

         match bound {
            Bound::Upper => line += " upperbound",
            Bound::Lower => line += " lowerbound",
            Bound::Exact => (),
         }
      }

      line += &format!(" nodes {}", curr.node);
      line += &format!(" time {:.0}", curr.time * 1E3);
      if let Some(speed) = curr.speed { line += &format!(" nps {:.0}", speed) }

      if !best.pv.is_empty() { line += &format!(" pv {}", game::line_to_uci(&best.pv, &self.board)) }

      println!("{}", line);
   }

   fn disp_time(&self, curr: &Current) {

      let mut line = "info".to_owned();
      line += &format!(" depth {}", curr.depth);
      if curr.ply_max != Ply(0) { line += &format!(" seldepth {}", curr.ply_max) }
      line += &format!(" nodes {}", curr.node);
      line += &format!(" time {:.0}", curr.time * 1E3);
      if let Some(speed) = curr.speed { line += &format!(" nps {:.0}", speed) }

      println!("{}", line);
   }

   fn ply_max(&self, ply_max: Ply) -> Ply {
      Ply(self.shared.ply_max.load(Relaxed)).max(ply_max)
   }

   fn node(&self, node: Node) -> Node {
      self.shared.node.load(Relaxed) + node
   }

   fn time(&self) -> Time {
      util::elapsed(&self.start)
   }

   fn alloc(&self) -> Time {
      let alloc = self.alloc_tc.unwrap();
      (alloc.time * self.factor()).min(alloc.max)
   }

   fn factor(&self) -> f32 {
      self.shared.factor.load()
   }

   fn is_smp(&self) -> bool {
      self.input.thread > 1
   }
}

impl<'s> SL<'s> {

   pub fn new(id: ID, sg: &'s SG<'_>) -> Self {

      let seed = sg.board.global.next_seed();

      Self {

         id,
         sg,

         rand: random::Random_Xor::from_seed(seed),

         best: Default::default(),
         curr: Default::default(),

         depth:   Ply(0),
         ply:     Ply(0),
         ply_max: Ply(0),
         period:  1, // root

         board: sg.board.clone(),
         list:  Vec::new(),

         todo: Stack_2::new(),
         score:Stack_2::new(),
         done: Stack_2::new(),
         dup:  Stack_2::new(),
         bad:  Stack_2::new(),

         index:  [0;                    Ply_Size as usize],
         move_:  [None;                 Ply_Size as usize],
         eval:   [None;                 Ply_Size as usize],
         pv:     [const { Vec::new() }; Ply_Size as usize],
         killer: util::array::from_clone(&Killer::new()),

         hist_0: Default::default(),
         hist_1: Default::default(),
         hist_2: Default::default(),

         eval_cache: Eval_Cache::new(16),
      }
   }

#[must_use]
   fn run(mut self) -> Output {

      // root moves

      if self.sg.is_smp() {

         for i in 0 .. 2 {
            for j in 0 .. 2 {
               for k in 0 .. Index_PT::Size {
                  self.hist_0[i][j][k].rand(&mut self.rand);
               }
            }
         }
      }

      self.gen_root(self.sg.pv_move);

      // iterative deepening

      for d in 1 ..= self.sg.limit_depth.val() {

         self.depth   = Ply(d);
         self.ply_max = Ply(0);

         self.curr.depth = self.depth;

         {
            let mut main = self.sg.main.lock().unwrap();
            main.start_iter(&self.sg, self.depth);
         }

         match self.search_asp(self.depth.into()) {
            Ok(())     => (),
            Err(Abort) => break,
         };

         {
            let mut main = self.sg.main.lock().unwrap();

            self.sg.update(&mut self.curr, self.ply_max, self.period);
            main.update(&self.sg, &self.best, &self.curr, true);

            main.end_iter(&self.sg, self.depth, self.best.move_);
         }

         if self.sg.input.tc.is_some()
         && self.depth == Ply(self.sg.shared.depth.load(Relaxed))
         {
            let used  = self.sg.time();
            let limit = self.sg.alloc();

            let phase  = eval::phase_side(&self.board, self.board.turn());
            let factor = util::math::lerp(0.5, 0.9, phase);

            if used >= limit * factor { break }
         }
      }

      debug_assert!(self.is_root());

      self.sg.shared.ply_max.fetch_max(self.ply_max.val(), Relaxed);
      self.sg.shared.node   .fetch_add(self.period, Relaxed);

      self.sg.shared.abort.store(true, Relaxed);

      Output::new(self.best, self.curr)
   }

   fn search_asp(&mut self, depth: Depth) -> Result<()> {

      let asp_depth  = Depth(4);
      let asp_margin = 15;

      if depth > asp_depth
      && let Some((mut score, bound)) = self.best.score
      {
         if score.is_loss() {

            debug_assert!(bound == Bound::Exact);

            loop {

               let ab = AB::new(score.dec(), score.inc());
               let sc = self.search_root(ab, depth)?;

               if sc <= ab.alpha && sc > Score::Min {
                  score = sc;
               } else if sc >= ab.beta {
                  self.search_root(AB::full(), depth)?;
                  return Ok(());
               } else {
                  return Ok(());
               }
            }

         } else if score.is_win() {

            debug_assert!(bound == Bound::Exact);

            loop {

               let ab = AB::new(score.dec(), score.inc());
               let sc = self.search_root(ab, depth)?;

               if sc >= ab.beta && sc < Score::Max {
                  score = sc;
               } else if sc <= ab.alpha {
                  self.search_root(AB::full(), depth)?;
                  return Ok(());
               } else {
                  return Ok(());
               }
            }

         } else if score.is_eval() {

            debug_assert!(bound == Bound::Exact);

            let mut ma = asp_margin;
            let mut mb = asp_margin;

            loop {

               debug_assert!(ma > 0);
               debug_assert!(mb > 0);

               let ab = AB::new((score - ma).max(Score::Min),
                                (score + mb).min(Score::Max));
               let sc = self.search_root(ab, depth)?;

               if sc <= ab.alpha && sc > Score::Min {

                  debug_assert!(ma <= 400);
                  ma *= 2;
                  if ma > 400 || !sc.is_eval() { ma = 20_000 }

               } else if sc >= ab.beta && sc < Score::Max {

                  debug_assert!(mb <= 400);
                  mb *= 2;
                  if mb > 400 || !sc.is_eval() { mb = 20_000 }

               } else {

                  return Ok(());
               }
            }

         } else {

            self.search_root(AB::full(), depth)?;
            Ok(())
         }

      } else {

         self.search_root(AB::full(), depth)?;
         Ok(())
      }
   }

   fn search_root(&mut self, ab: AB, depth: Depth) -> Result<Score> {
      let local = &mut Local::new(ab, depth, true);
      self.search_root_node(local)
   }

   fn search_root_node(&mut self, local: &mut Local) -> Result<Score> {

      debug_assert!(local.depth > Depth(0));
      debug_assert!(self.is_root());

      self.pv[self.ply.index()].clear();

      local.pv_node = local.pv_node && local.depth > Depth(4);

      let eval = self.eval();

      local.li   = Some(Legal_Info::new(&self.board));
      local.eval = Some(eval);

      // move loop

      self.done.clear();

      for i in self.list.range() {

         let bd = &self.board;
         let mv = self.list[i as usize].normalise(bd);
         let mi = &mut Move_Info::new(mv, bd, local.in_check(), Stage::Quiet);

         // extension/reduction

         let ext = self.extend(mv, mi, local);
         let red = self.reduce(mv, mi, local);

         let ab = local.ab.update(local.score);
         let d  = local.depth + ext - Depth(1);

         let first = self.done.size() == 0;
         let undo  = &self.push(Some(mv))?; // affects self.done #

         let mut child;

         if (local.pv_node && !first) || red != Depth(0) {

            child = self.search(-AB::new(ab.alpha, ab.alpha.inc()), d - red, false);

            if let Ok(score) = child && -score > ab.alpha {
               child = self.search(-ab, d, local.pv_node);
            }

         } else {

            child = self.search(-ab, d - red, local.pv_node);
         }

         self.pop(Some(mv), undo);

         let sc = -child?;

         if sc > ab.alpha { local.move_ = Some(Move_Long::new(mv, &self.board)) }
         local.score = local.score.max(sc);

         if first || sc > ab.alpha { // new best move

            self.concat_pv(mv);

            self.best = Best {
               move_: Some(mv),
               score: Some((sc, Bound::new(sc, local.ab))),
               depth: self.depth,
               pv:    self.pv[self.ply.index()].clone(),
            };

            if self.depth > Ply(1) {

               let mut main = self.sg.main.lock().unwrap();

               self.sg.update(&mut self.curr, self.ply_max, self.period);
               main.update(&self.sg, &self.best, &self.curr, !first);
            }

            if self.list[i as usize] == mv { // not always true?
               self.list.move_to_front(i);
            }
         }

         let mv_32 = Move_Long::new(mv, &self.board);
         debug_assert!(!self.done.has(mv_32));
         self.done.add(mv_32);

         if sc >= local.ab.beta { break }
      }

      debug_assert!(local.score.is_valid());
      debug_assert!(self.best.depth == self.depth);

      self.tt_store(self.board.key(), local);

      Ok(local.score)
   }

   fn search(&mut self, ab: AB, depth: Depth, pv_node: bool) -> Result<Score> {
      let local = &mut Local::new(ab, depth, pv_node);
      self.search_node(local)
   }

   fn search_node(&mut self, local: &mut Local) -> Result<Score> {

      debug_assert!(self.ply <= self.sg.limit_ply);

      self.pv[self.ply.index()].clear();

      local.pv_node = local.pv_node && local.depth > Depth(4);

      let bd = &self.board;
      debug_assert!(!bd.is_illegal());

      // leaf

      let min = self.score_loss(self.ply);
      let max = self.score_win (self.ply + Ply(1));

      if min >= local.ab.beta  { return Ok(min) }
      if max <= local.ab.alpha { return Ok(max) }

      if local.depth <= Depth(0)       { return Ok(self.qs(local.ab)?) }
      if bd.rep().is_some()            { return Ok(self.score_rep()) }
      if self.ply >= self.sg.limit_ply { return Ok(self.eval()) }

      // transposition table

      local.tt_depth  = Depth(0);
      local.tt_score  = (-Score::Inf, Score::Inf);
      let mut tt_eval = None;

      local.tt_key = Some(bd.key());

      if let Some(to) = self.sg.tt.probe(local.tt_key.unwrap()) {

         let (mut min, mut max) = (to.min, to.max);

         if min != -Score::Inf { min = min.add_ply(self.ply) }
         if max !=  Score::Inf { max = max.add_ply(self.ply) }

         if !local.pv_node {

            if min == max && to.d_min.min(to.d_max) >= local.depth.val() {
               return Ok(min);
            }

            if min >= local.ab.beta  && to.d_min >= local.depth.val() { return Ok(min) }
            if max <= local.ab.alpha && to.d_max >= local.depth.val() { return Ok(max) }
         }

         local.tt_depth = Depth::from_int(to.d_min);
         local.tt_score = (min, max);

         local.pv_move = to.move_;
         tt_eval = to.eval;
      }

      if let Some(eval) = tt_eval { self.set_eval(eval) }

      let eval = tt_eval.unwrap_or_else(|| self.eval());
      let bd = &self.board;

      local.li   = Some(Legal_Info::new(bd));
      local.eval = Some(eval);

      // stalemate

      if !local.in_check() {

         if !local.li().seem_legal(bd) && !bd.can_play() {
            return Ok(self.score_draw());
         }

         let min = self.score_loss(self.ply + Ply(2));
         if min >= local.ab.beta { return Ok(min) }
      }

      // eval pruning

      let ep_depth  = Depth(4);
      let ep_margin = 75;

      if local.depth <= ep_depth
      && !local.in_check()
      && !bd.is_lone_king(bd.turn())
      {
         let eval = local.eval.unwrap();

         let sc = eval - Score(local.depth * ep_margin);
         if sc >= local.ab.beta { return Ok(eval) }
      }

      // static null-move pruning

      let snmp_depth = Depth(3);
      let snmp_tempo = 10;

      if local.depth <= snmp_depth
      && !local.in_check()
      && !bd.is_lone_king(bd.turn())
      {
         self.board.pass();
         let sc = -self.qs_static(-AB::new(local.ab.beta.dec(), local.ab.beta), -(local.eval.unwrap() - snmp_tempo * 2));
         self.board.pass();

         if sc >= local.ab.beta { return Ok(sc) }
      }

      let bd = &self.board;

      // null-move pruning

      let nmp_depth = Depth(3);
      let nmp_red   = Depth(2) + local.depth.mul_shift(1, 2);

      if local.depth > nmp_depth
      && !local.in_check()
      && !local.pv_node
      && local.ab.beta.is_eval()
      && local.eval.unwrap() >= local.ab.beta
      && !bd.is_lone_king(bd.turn())
      {
         let undo = &self.push(None)?;
         let child = self.search(-AB::new(local.ab.beta.dec(), local.ab.beta), local.depth - nmp_red - Depth(1), false);
         self.pop(None, undo);

         let mut sc = -child?;

         if sc >= local.ab.beta {

            sc = sc.min(Score::Eval_Max);
            debug_assert!(sc >= local.ab.beta);

            if sc >= local.ab.beta { return Ok(sc) }
         }
      }

      // ProbCut

      let pc_depth  = Depth(6);
      let pc_margin = Score(150);

      if local.depth >= pc_depth
      && !local.pv_node
      && local.ab.beta.is_eval()
      {
         let beta = local.ab.beta + pc_margin;
         let red  = local.depth.mul_shift(1, 1).min(Depth(4));

         let _ = self.gen_capture(local.li());
         self.done.clear();

         for i in self.todo.range() {

            let mv = self.todo[i].normalise(&self.board);

            if mv.is_win(&self.board)
            && local.li().is_legal(mv, &self.board)
            {
               let undo = &self.push(Some(mv))?;
               let child = self.search(-AB::new(beta.dec(), beta), local.depth - red - Depth(1), local.pv_node);
               self.pop(Some(mv), undo);

               let sc = -child?;
               if sc >= beta { return Ok(sc) }
            }
         }
      }

      // futility pruning

      local.fp_node  = false;
      local.fp_score = None;

      let fp_depth  = Depth(4);
      let fp_margin = 30;

      if local.depth <= fp_depth {

         let sc = local.eval.unwrap() + Score(local.depth * fp_margin);

         if sc <= local.ab.alpha {

            local.score = sc; // represent skipped moves
            local.skip  = true;

            local.fp_node  = true;
            local.fp_score = Some(sc);
         }
      }

      if !local.fp_node
      && local.depth <= Depth(3)
      && !local.in_check()
      {
         local.skip = true;
      }

      // move loop

      self.todo .clear();
      self.score.clear();
      self.done .clear();
      self.dup  .clear();
      self.bad  .clear();

      if local.in_check() {
        self.search_eva(local)?;
      } else if local.fp_node {
         self.search_fp(local)?;
      } else {
         self.search_steps(local)?;
      }

      if local.score.is_loss() && local.skip && local.score <= local.ab.alpha {

         let sc = if local.score == -Score::Inf && !local.in_check() && !local.li().seem_legal(&self.board) {
            self.score_draw()
         } else {
            local.ab.alpha.max(self.score_loss(self.ply + Ply(2)))
         };

         return Ok(sc);

      } else if local.score == -Score::Inf { // no moves => game end

         return Ok(self.score_end(local.in_check()));
      }

      debug_assert!(local.score >= self.score_loss(self.ply));
      debug_assert!(local.score <= self.score_win (self.ply + Ply(1)));

      self.tt_store(local.tt_key.unwrap(), local);

      if let Some(good) = local.move_
      && !(good.is_cap() || good.is_prom())
      {
         if !local.in_check() {
            self.killer[self.ply.index()].add(good);
         }

         let turn    = self.board.turn().index();
         let index_1 = if self.ply >= Ply(1) { self.index[(self.ply - Ply(1)).index()] } else { 0 };
         let index_2 = if self.ply >= Ply(2) { self.index[(self.ply - Ply(2)).index()] } else { 0 };

         let hist_0 = &mut self.hist_0[turn][local.in_check() as usize];
         let hist_1 = &mut self.hist_1[turn][index_1];
         let hist_2 = &mut self.hist_2[turn][index_2];

         let d = u16::from(local.depth.val());
         let bonus = (d * 40).min(2_000);

         hist_0[good].good(bonus, 12);
         hist_1[good].good(bonus, 12);
         hist_2[good].good(bonus, 12);

         debug_assert!(self.done.has(good));

         for &bad in self.done.iter() {

            if bad == good { break }

            if !(bad.is_cap() || bad.is_prom()) {
               hist_0[bad].bad(bonus, 12);
               hist_1[bad].bad(bonus, 12);
               hist_2[bad].bad(bonus, 12);
            }
         }
      }

      Ok(local.score)
   }

   fn search_steps(&mut self, local: &mut Local) -> Result<()> {

      debug_assert!(!local.in_check());
      debug_assert!(!local.fp_node);

      local.stage = self.gen_pv(local.li(), local.pv_move, false);
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_capture(local.li());
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_promotion(local.li());
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_killer(local.li());
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_quiet(local.li(), local.depth);
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_bad();
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      Ok(())
   }

   fn search_eva(&mut self, local: &mut Local) -> Result<()> {

      debug_assert!(local.in_check());

      local.stage = self.gen_pv(local.li(), local.pv_move, false);
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_capture(local.li());
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      if !local.fp_node {
         local.stage = self.gen_quiet(local.li(), local.depth);
         if let Err(Return) = self.search_list(local)? { return Ok(()) }
      }

      local.stage = self.gen_bad();
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      Ok(())
   }

   fn search_fp(&mut self, local: &mut Local) -> Result<()> {

      debug_assert!(local.fp_node);
      debug_assert!(!local.li().in_check());

      local.stage = self.gen_pv(local.li(), local.pv_move, true);
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_capture(local.li());
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      gen_::add_proms(&mut self.todo, &self.board);
      local.stage = Stage::Promotion;
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      gen_::add_castle(&mut self.todo, &self.board);
      local.stage = Stage::Quiet;
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      gen_::add_push(&mut self.todo, &self.board);
      local.stage = Stage::Quiet;
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      gen_::add_checks_fast(&mut self.todo, &self.board);
      local.stage = Stage::Quiet;
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      local.stage = self.gen_bad();
      if let Err(Return) = self.search_list(local)? { return Ok(()) }

      Ok(())
   }

   fn search_list(&mut self, local: &mut Local) -> Result<std::result::Result<(), Return>> {

      for i in self.todo.range() {

         let mv = self.todo[i].normalise(&self.board);

         self.search_move(local, mv)?;
         if local.stop() { return Ok(Err(Return)) }
      }

      self.todo .clear();
      self.score.clear();

      Ok(Ok(()))
   }

   fn search_move(&mut self, local: &mut Local, mv: Move) -> Result<()> {

      debug_assert!(!local.stop());

      let bd = &self.board;

      if self.dup.has(mv) { return Ok(()) }

      if local.stage == Stage::Quiet
      && mv.is_prom(bd)
      && !local.in_check()
      {
         return Ok(());
      }

      if !local.li().is_legal(mv, bd) { return Ok(()) }

      let mi = &mut Move_Info::new(mv, bd, local.in_check(), if !local.fp_node { local.stage } else { Stage::Quiet });

      if !local.fp_node
      && (mi.is_cap() || mi.is_prom())
      && Some(mv) != local.pv_move
      && local.stage != Stage::Bad
      && !mi.is_safe(bd)
      {
         self.bad.add(mv);
         return Ok(());
      }

      if !local.fp_node
      && local.depth <= Depth(3)
      && !(mi.is_cap() || mi.is_prom())
      && !mi.is_safe(bd)
      {
         local.skip = true;
         return Ok(());
      }

      if local.fp_node
      && local.depth <= Depth(4)
      && !mi.is_check(bd)
      && local.fp_score.unwrap() + see::max(mv, bd) + Score(local.depth * 40) <= local.ab.alpha
      {
         return Ok(());
      }

      if local.fp_node
      && local.depth <= Depth(2)
      && !mi.is_safe(bd)
      {
         return Ok(());
      }

      // extension/reduction

      let ext = self.extend(mv, mi, local);
      let mut red = self.reduce(mv, mi, local);

      let ab = local.ab.update(local.score);
      let d  = local.depth + ext - Depth(1);

      if d - red <= Depth(0) && local.depth > Depth(1) {
         red = d - Depth(1);
         debug_assert!(red >= Depth(0));
      }

      let first = self.done.size() == 0;
      let undo  = &self.push(Some(mv))?; // affects self.done #

      let mut child;

      if (local.pv_node && !first) || red != Depth(0) {

         child = self.search(-AB::new(ab.alpha, ab.alpha.inc()), d - red, false);

         if let Ok(score) = child && -score > ab.alpha {
            child = self.search(-ab, d, local.pv_node);
         }

      } else {

         child = self.search(-ab, d - red, local.pv_node);
      }

      self.pop(Some(mv), undo);

      let mv_32 = Move_Long::new(mv, &self.board);
      let sc = -child?;

      if sc > ab.alpha {
         local.move_ = Some(mv_32);
         if sc < local.ab.beta { self.concat_pv(mv) }
      }

      local.score = local.score.max(sc);

      debug_assert!(!self.done.has(mv_32));
      self.done.add(mv_32);

      debug_assert!(!self.dup.has(mv));
      if matches!(local.stage, Stage::PV | Stage::Promotion | Stage::Killer) {
         self.dup.add(mv);
      }

      // late-move pruning

      let lmp_depth = Depth(2);
      let lmp_late  = 4;

      if self.done.size() >= (local.depth * lmp_late) as u16
      && local.depth <= lmp_depth
      {
         local.skip = true;
         local.stop = true;
      }

      // underpromotion

      if mv.auto_queen(&self.board) == Some(Piece::Queen)
      && !local.fp_node
      && local.depth > Depth(4)
      {
         let from = mv.from();
         let to   = mv.to();

         self.bad.add(Move::with_prom(from, to, Piece::Rook));
         self.bad.add(Move::with_prom(from, to, Piece::Knight));
         self.bad.add(Move::with_prom(from, to, Piece::Bishop));
      }

      Ok(())
   }

   fn qs(&mut self, ab: AB) -> Result<Score> {

      let bd = &self.board;

      debug_assert!(!bd.is_illegal());
      // debug_assert!(!bd.in_check());

      self.qs_check(ab, Ply(1))
   }

   fn qs_check(&mut self, ab: AB, check: Ply) -> Result<Score> {

      debug_assert!(self.ply <= self.sg.limit_ply);

      let bd = &self.board;
      debug_assert!(!bd.is_illegal());

      // leaf

      if bd.rep().is_some() { return Ok(self.score_rep()) }

      if check == Ply(0) { return self.qs_cap(ab, Ply(1)) } // capture-only

      if self.ply >= self.sg.limit_ply { return Ok(self.eval()) }

      // init

      let mut move_ = None;
      let mut score = -Score::Inf;
      let mut eval  = None;
      let mut skip  = false;

      // transposition table

      let tt_key = bd.key();
      let mut pv_move = None;

      if let Some(to) = self.sg.tt.probe(tt_key) {

         let (mut min, mut max) = (to.min, to.max);

         if min != -Score::Inf { min = min.add_ply(self.ply) }
         if max !=  Score::Inf { max = max.add_ply(self.ply) }

         if min == max      { return Ok(min) }
         if min >= ab.beta  { return Ok(min) }
         if max <= ab.alpha { return Ok(max) }

         pv_move = to.move_;
         eval = to.eval;
      }

      if let Some(eval) = eval { self.set_eval(eval) }

      // move generation

      let bd = &self.board;
      let li = &Legal_Info::new(bd);

      if li.in_check() { // generate all moves

         if eval.is_none() { eval = Some(self.eval()) }
         let bd = &self.board;

         let list = &mut self.todo;
         gen_::gen_eva_caps  (list, bd, li.checks());
         order::sort_caps    (list, bd);
         gen_::add_eva_quiets(list, bd, li.checks());

      } else { // stand pat, then generate tactical moves

         score = eval.unwrap_or_else(|| self.eval());
         eval  = Some(score);
         skip  = true;

         if score >= ab.beta { return Ok(score) } // stand pat

         self.gen_qs(Some(li), true);
      }

      if let Some(good) = pv_move { order::move_to_front(&mut self.todo, good) }

      // move loop

      for i in self.todo.range() {

         let bd = &self.board;
         let mv = self.todo[i].normalise(bd);

         debug_assert!(!mv.is_king_cap(bd));

         if !li.is_legal(mv, bd) { continue }

         let move_is_check = !li.in_check() && mv.is_check(bd);

         if !move_is_check
         && eval.unwrap() + see::max(mv, bd) <= ab.alpha
         {
            skip = true;
            continue;
         }

         if !mv.is_safe(bd) {
            skip = true;
            continue;
         }

         let ext = Ply(if move_is_check { 1 } else { 0 });

         let undo = &self.push(Some(mv))?;
         let child = self.qs_check(-ab.update(score), check + ext - Ply(1));
         self.pop(Some(mv), undo);

         let sc = -child?;

         if sc > ab.update(score).alpha { move_ = Some(mv) }
         score = score.max(sc);

         if sc >= ab.beta { break }
      }

      if score.is_loss() && skip && score <= ab.alpha {
         return Ok(ab.alpha.max(self.score_loss(self.ply + Ply(2))));
      } else if score == -Score::Inf { // no moves => game end
         debug_assert!(li.in_check());
         return Ok(self.score_loss(self.ply));
      }

      debug_assert!(score >= self.score_loss(self.ply));
      debug_assert!(score <= self.score_win (self.ply + Ply(1)));

      let (mut min, mut max) = ab.score_to_range(score);

      if min != -Score::Inf { min = min.sub_ply(self.ply) }
      if max !=  Score::Inf { max = max.sub_ply(self.ply) }

      self.sg.tt.store(tt_key, move_, min, max, eval, 0, false);

      Ok(score)
   }

   fn qs_cap(&mut self, ab: AB, ply: Ply) -> Result<Score> {

      debug_assert!(self.ply <= self.sg.limit_ply);

      // leaf

      let eval = self.eval();
      let mut score = eval;

      if score >= ab.beta || self.ply >= self.sg.limit_ply { // stand pat
         return Ok(score);
      }

      // move loop

      if ply >= Ply(4) {
         gen_::gen_recaps(&mut self.todo, &self.board);
      } else {
         self.gen_qs(None, false);
      }

      for i in self.todo.range() {

         let bd = &self.board;
         let mv = self.todo[i].normalise(bd);

         if mv.is_king_cap(bd) { // king capture => illegal position
            return Ok(self.score_win(self.ply - Ply(1)));
         }

         if eval + see::max(mv, bd) <= ab.alpha { continue }
         if !mv.is_safe(bd) { continue }

         let undo = &self.push(Some(mv))?;
         let child = self.qs_cap(-ab.update(score), ply + Ply(1));
         self.pop(Some(mv), undo);

         let sc = -child?;
         score = score.max(sc);

         if sc >= ab.beta { break }
      }

      debug_assert!(score >= self.score_loss(self.ply));
      debug_assert!(score <= self.score_win (self.ply + Ply(1)));

      Ok(score)
   }

   fn qs_static(&mut self, ab: AB, eval: Score) -> Score {

      // leaf

      let mut score = eval;
      if score >= ab.beta { return score } // stand pat

      // move loop

      let list = &mut self.todo;
      let bd   = &self.board;

      gen_::gen_caps (list, bd);
      gen_::add_proms(list, bd);

      for &mv in list.iter() {

         let sc = eval + see::see_move(mv, bd);

         score = score.max(sc);
         if sc >= ab.beta { break }
      }

      debug_assert!(score.is_eval());
      score
   }

   fn extend(&self, mv: Move, mi: &mut Move_Info, local: &Local) -> Depth {

      let bd = &self.board;

      if local.pv_node {

         if mi.is_check(bd) || mi.is_recap(bd) {
            return Depth(1);
         }

      } else if local.depth <= Depth(4) {

         if (mi.is_check(bd) && mi.is_safe  (bd))
         || (mi.is_recap(bd) && mv.is_unique(bd))
         {
            return Depth(1);
         }
      }

      Depth(0)
   }

   fn reduce(&self, mv: Move, mi: &mut Move_Info, local: &Local) -> Depth {

      let lmr_depth = Depth(3);
      let lmr_late  = 2;

      let mut red = Depth(0);
      let bd   = &self.board;
      let late = self.done.size();

      if local.depth > lmr_depth && late >= lmr_late {

         red = bd.global.table_search.lmr_red(local.depth, late);

         if local.pv_node { red -= Depth(1) }

         if mi.is_cap()      { red -= Depth(1) }
         if mi.is_check(bd)  { red -= Depth(1) }
         if local.in_check() { red -= Depth(1) }

         if mi.is_cap() && mi.is_safe(bd) { red -= Depth(1) }
      }

      red.max(Depth(0))
   }

   fn tt_store(&mut self, key: Key, local: &Local) {

      debug_assert!(local.score.is_valid());

      let (mut min, mut max) = local.ab.score_to_range(local.score);

      if min != -Score::Inf { min = min.sub_ply(self.ply) }
      if max !=  Score::Inf { max = max.sub_ply(self.ply) }

      self.sg.tt.store(self.board.key(), local.move_.map(|mv| mv.into()), min, max, local.eval, local.depth.val(), local.pv_node);
   }

   fn concat_pv(&mut self, mv: Move) {

      self.pv[self.ply.index()].clear();
      self.pv[self.ply.index()].push(mv);

      for i in 0 .. self.pv[(self.ply + Ply(1)).index()].len() {
         let mv = self.pv[(self.ply + Ply(1)).index()][i];
         self.pv[self.ply.index()].push(mv);
      }
   }

   fn gen_root(&mut self, pv_move: Option<Move>) {

      debug_assert!(self.is_root());

      let li = &Legal_Info::new(&self.board);

      let stage = self.gen_all(li, pv_move);
      assert!(stage == Stage::All);

      // filter legal moves

      assert!(self.list.is_empty());
      self.list.clear();

      for &mv in self.todo.iter() {

         if li.is_legal(mv, &self.board) {

            if mv.is_prom(&self.board) {

               let from = mv.from();
               let to   = mv.to();

               self.list.push(Move::with_prom(from, to, Piece::Queen));
               self.list.push(Move::with_prom(from, to, Piece::Rook));
               self.list.push(Move::with_prom(from, to, Piece::Knight));
               self.list.push(Move::with_prom(from, to, Piece::Bishop));

            } else {

               self.list.push(mv);
            }
         }
      }
   }

#[must_use]
   fn gen_all(&mut self, li: &Legal_Info, pv_move: Option<Move>) -> Stage {

      debug_assert!(self.is_root());

      let list = &mut self.todo;
      let bd   = &self.board;

      let split;

      if li.in_check() {

         gen_::gen_eva_caps(list, bd, li.checks());
         order::sort_caps  (list, bd);
         split = list.len();
         gen_::add_eva_quiets(list, bd, li.checks());

      } else {

         gen_::gen_caps  (list, bd);
         order::sort_caps(list, bd);
         split = list.len();
         gen_::add_quiets(list, bd);
      }

      let index_1 = if self.ply >= Ply(1) { self.index[(self.ply - Ply(1)).index()] } else { 0 };
      let index_2 = if self.ply >= Ply(2) { self.index[(self.ply - Ply(2)).index()] } else { 0 };

      let hist_0 = &self.hist_0[bd.turn().index()][li.in_check() as usize];
      let hist_1 = &self.hist_1[bd.turn().index()][index_1];
      let hist_2 = &self.hist_2[bd.turn().index()][index_2];

      self.score.clear();

      for _ in split .. self.todo.size() as usize {
         self.score.add(0);
      }

      order::sort_quiets(&mut self.todo.as_mut_slice()[split ..], &mut self.score, bd, hist_0, hist_1, hist_2);

      if let Some(good) = pv_move { order::move_to_front(&mut self.todo, good) }

      Stage::All
   }

#[must_use]
   fn gen_pv(&mut self, li: &Legal_Info, pv_move: Option<Move>, fp_node: bool) -> Stage {

      if let Some(mv) = pv_move && li.is_pseudo(mv, &self.board) {
         self.todo.add(mv);
      }

      Stage::PV
   }

#[must_use]
   fn gen_capture(&mut self, li: &Legal_Info) -> Stage {

      let list = &mut self.todo;
      let bd   = &self.board;

      if li.in_check() {
         gen_::gen_eva_caps(list, bd, li.checks());
      } else {
         gen_::gen_caps(list, bd);
      }

      order::sort_caps(list, bd);

      Stage::Capture
   }

#[must_use]
   fn gen_promotion(&mut self, li: &Legal_Info) -> Stage {
      if !li.in_check() { gen_::add_proms(&mut self.todo, &self.board) }
      Stage::Promotion
   }

#[must_use]
   fn gen_killer(&mut self, li: &Legal_Info) -> Stage {

      if !li.in_check() {

         for mv in self.killer[self.ply.index()].iter() {

            if li.long_is_pseudo(mv, &self.board)
            && !(mv.is_cap() || mv.is_prom())
            {
               self.todo.add(mv.into());
            }
         }
      }

      Stage::Killer
   }

#[must_use]
   fn gen_quiet(&mut self, li: &Legal_Info, depth: Depth) -> Stage {

      let list = &mut self.todo;
      let bd   = &self.board;

      if li.in_check() {
         gen_::add_eva_quiets(list, bd, li.checks());
      } else if depth <= Depth(3) {
         gen_::add_safe(list, bd);
      } else {
         gen_::add_quiets(list, bd);
      }

      let index_1 = if self.ply >= Ply(1) { self.index[(self.ply - Ply(1)).index()] } else { 0 };
      let index_2 = if self.ply >= Ply(2) { self.index[(self.ply - Ply(2)).index()] } else { 0 };

      let hist_0 = &self.hist_0[bd.turn().index()][li.in_check() as usize];
      let hist_1 = &self.hist_1[bd.turn().index()][index_1];
      let hist_2 = &self.hist_2[bd.turn().index()][index_2];

      self.score.clear();

      for i in self.todo.range() {
         self.score.add(0);
      }

      order::sort_quiets(&mut self.todo, &mut self.score, bd, hist_0, hist_1, hist_2);

      Stage::Quiet
   }

#[must_use]
   fn gen_bad(&mut self) -> Stage {

      self.todo.copy(self.bad.iter().copied());
      self.bad.clear();

      Stage::Bad
   }

   fn gen_qs(&mut self, li: Option<&Legal_Info>, check: bool) {

      let list = &mut self.todo;
      let bd   = &self.board;

      gen_::gen_caps  (list, bd);
      order::sort_caps(list, bd);
      gen_::add_proms (list, bd);
      gen_::add_castle(list, bd);
      gen_::add_push  (list, bd);
      if check { gen_::add_checks_fast(list, bd) }
   }

   fn score_end(&self, in_check: bool) -> Score {

      let bd = &self.board;
      debug_assert!(!bd.can_play());

      if !in_check {
         self.score_draw()
      } else {
         self.score_loss(self.ply)
      }
   }

   fn score_win(&self, ply: Ply) -> Score {
      Score::Max.add_ply(ply)
   }

   fn score_loss(&self, ply: Ply) -> Score {
      Score::Min.add_ply(ply)
   }

   fn score_rep(&mut self) -> Score {
      self.score_draw() + self.rand()
   }

   fn score_draw(&self) -> Score {
      let sc = Score(self.board.global.score_draw.load(Relaxed));
      if self.board.turn() != self.sg.board.turn() { -sc } else { sc }
   }

   fn eval(&mut self) -> Score { // for 'turn'

      if let Some(old) = self.eval[self.ply.index()] {
         old
      } else {
         let new = self.comp_eval();
         self.eval[self.ply.index()] = Some(new);
         new
      }
   }

   fn set_eval(&mut self, eval: Score) {
      self.eval[self.ply.index()] = Some(eval);
   }

   fn comp_eval(&mut self) -> Score { // for 'turn'

      let bd = &self.board;
      let sd = bd.turn();
      let xd = sd.opp();

      let key = bd.key();

      let eval;

      if let Some(e) = self.eval_cache.probe(key) {

         eval = e;

      } else {

         let sc = self.board.global.eval.eval(bd);
         eval = sd.sign(sc.clamp(Score::Eval_Min, Score::Eval_Max));
         self.eval_cache.store(key, eval);
      }

      eval
   }

   fn rand(&mut self) -> Score {

      let size = 3;
      let half = 1 << (size - 1);

      let rand = loop {
         let rand = self.rand.next_bits(size) as i16;
         if rand != 0 { break rand }
      };

      Score(rand - half)
   }

#[must_use]
   fn push(&mut self, move_: Option<Move>) -> Result<Frame> {

      debug_assert!(self.ply < self.sg.limit_ply);

      self.inc_node()?;

      let undo = if let Some(mv) = move_ {

         self.index[self.ply.index()] = Index_PT::index_16 (mv, &self.board);
         self.move_[self.ply.index()] = Some(Move_Long::new(mv, &self.board));

         self.board.do_move(mv)

      } else {

         self.index[self.ply.index()] = 0;
         self.move_[self.ply.index()] = None;

         self.board.do_null()
      };

      self.ply += Ply(1);
      self.ply_max = self.ply_max.max(self.ply);

      let fp_todo  = self.todo .enter();
      let fp_score = self.score.enter();
      let fp_done  = self.done .enter();
      let fp_dup   = self.dup  .enter();
      let fp_bad   = self.bad  .enter();

      Ok(Frame { undo, fp_todo, fp_score, fp_done, fp_dup, fp_bad })
   }

   fn pop(&mut self, move_: Option<Move>, frame: &Frame) {

      debug_assert!(!self.is_root());

      let Frame { undo, fp_todo, fp_score, fp_done, fp_dup, fp_bad } = frame;

      if let Some(mv) = move_ {
         self.board.undo_move(mv, undo);
      } else {
         self.board.undo_null(undo);
      }

      self.todo .leave(*fp_todo);
      self.score.leave(*fp_score);
      self.done .leave(*fp_done);
      self.dup  .leave(*fp_dup);
      self.bad  .leave(*fp_bad);

      self.eval[self.ply.index()] = None;

      self.killer[self.ply.index() + 1].clear();

      self.ply -= Ply(1);
   }

   fn inc_node(&mut self) -> Result<()> {

      if self.period == 1_000 {

         // update shared statistics

         self.sg.shared.ply_max.fetch_max(self.ply_max.val(), Relaxed);
         let sg_node = self.sg.shared.node.fetch_add(self.period, Relaxed) + self.period;

         self.period = 0;

         // abort search?

         if self.sg.shared.abort.load(Relaxed) {
            return Err(Abort);
         }

         if let Some(node) = self.sg.limit_node
         && sg_node >= node
         && self.depth > Ply(1)
         {
            return Err(Abort);
         }

         if let Some(time) = self.sg.limit_time
         && self.sg.time() >= time
         && self.depth > Ply(1)
         {
            return Err(Abort);
         }

         if let Some(alloc) = self.sg.alloc_tc
         && self.sg.time() >= self.sg.alloc()
         && self.depth > Ply(1)
         {
            return Err(Abort);
         }

         if self.id == 0 && let Some(line) = self.board.global.input.peek() {

            if line == Some("isready".to_owned()) {
               println!("readyok");
               self.board.global.input.get(); // skip input line
            } else {
               return Err(Abort);
            }
         }
      }

      self.period += 1;

      Ok(())
   }

   fn is_root(&self) -> bool {
      self.ply == Ply(0)
   }
}

impl Default for Input {

   fn default() -> Self {

      Self {

         thread: 1,

         depth: None,
         ply:   None,
         node:  None,
         time:  None,
         tc:    None,

         score: false,
      }
   }
}

impl Output {

   fn new(best: Best, curr: Current) -> Self {
      Self { best, curr }
   }
}

impl Default for Best {

   fn default() -> Self {

      Self {
         move_: None,
         score: None,
         depth: Ply(0),
         pv:    vec![],
      }
   }
}

impl Default for Current {

   fn default() -> Self {

      Self {
         depth:   Ply(0),
         ply_max: Ply(0),
         node:    1, // root
         time:    0.0,
         speed:   None,
      }
   }
}

impl Shared {

   fn new() -> Self {

      Self {

         ply_max: 0.into(),
         node:    0.into(),

         factor: 1.0  .into(),
         abort:  false.into(),
         depth:  0    .into(),
      }
   }
}

impl Main {

   fn new() -> Self {

      Self {

         output:    Default::default(),
         last_best: Default::default(),

         depth:  Ply(0),
         factor: 1.0,
      }
   }

   fn start_iter(&mut self, sg: &SG<'_>, depth: Ply) {
      self.depth = self.depth.max(depth);
      sg.shared.depth.fetch_max(depth.val(), Relaxed);
   }

   fn end_iter(&mut self, sg: &SG<'_>, depth: Ply, mv: Option<Move>) {

      if depth == self.depth {

         if self.output.best.move_ == self.last_best.move_ {
            self.factor = (self.factor - 0.1).max(0.5);
         }

         self.last_best = self.output.best.clone();
      }
   }

   fn update(&mut self, sg: &SG<'_>, best: &Best, curr: &Current, new_move: bool) {

      let vry = &self.last_best;
      let old = &self.output.best;
      let new = best;

      let (s0, b0) = vry.score.unwrap_or((-Score::Inf, Bound::Lower));
      let (s1, b1) = old.score.unwrap_or((-Score::Inf, Bound::Lower));
      let (s2, b2) = new.score.unwrap();

      let (i0, a0) = b0.to_range(s0);
      let (i1, a1) = b1.to_range(s1);
      let (i2, a2) = b2.to_range(s2);

      let more_depth = new.depth > old.depth;
      let more_score = i2 > i1;
      let same_move  = new.move_ == old.move_;

      let more_info =  new.depth > old.depth
                    || (b2.is_lower() && s2 > s1)
                    || (b2 == Bound::Exact && b1 != Bound::Exact)
                    || (b2.is_upper() && b1 == Bound::Upper && s2 < s1)
                    ;

      if !sg.is_smp() || more_depth || more_score || (same_move && more_info) {

         if a2.val() - a0.val() <= -15 {
            self.factor = (self.factor.max(1.0) + 0.2).min(2.0);
         }

         self.output = Output::new(best.clone(), curr.clone());
         sg.disp_move(best, curr);

         sg.shared.factor.store(self.factor);
      }
   }
}

impl Local {

   fn new(ab: AB, depth: Depth, pv_node: bool) -> Self {

      Local {

         ab,
         depth,
         pv_node,

         move_: None,
         score: -Score::Inf,
         skip:  false,

         tt_key:   None,
         pv_move:  None,
         tt_depth: Depth(0),
         tt_score: (-Score::Inf, Score::Inf),

         li:   None,
         eval: None,

         fp_node:  false,
         fp_score: None,

         stage: Stage::Bad,
         stop:  false,
      }
   }

   fn li(&self) -> &Legal_Info {
      (&self.li).as_ref().unwrap()
   }

   fn in_check(&self) -> bool {
      self.li().in_check()
   }

   fn stop(&self) -> bool {
      self.score >= self.ab.beta || self.stop
   }
}

impl Table_Search {

   const Depth_Size: u8 = 64;
   const Late_Size:  u8 = 64;

   pub fn new() -> Self {

      let shift  = 1;
      let factor = 0.30;

      let mut lmr = [[0; Self::Late_Size as usize]; Self::Depth_Size as usize];

      for di in 1 .. Self::Depth_Size {

         let di = Depth::from_int(di);
         let dr = f32::from(di);

         for l in 1 .. Self::Late_Size {

            let rr = dr.log(2.0) * f32::from(l + shift).log(2.0) * factor;
            let ri = Depth::try_from(rr.max(0.0)).unwrap();

            lmr[di.val() as usize][l as usize] = ri.val();
         }
      }

      Self { lmr }
   }

   fn lmr_red(&self, depth: Depth, late: u16) -> Depth {

      let d = depth.val().min(Self::Depth_Size - 1);
      let l = late.min(u16::from(Self::Late_Size) - 1);

      Depth::from_int(self.lmr[d as usize][l as usize])
   }
}

impl AB {

   fn new(alpha: Score, beta: Score) -> Self {
      debug_assert!(Score::Min <= alpha && alpha < beta && beta <= Score::Max);
      Self { alpha, beta }
   }

   fn full() -> Self {
      Self::new(Score::Min, Score::Max)
   }

   fn update(self, sc: Score) -> Self {
      debug_assert!(sc < self.beta);
      Self::new(self.alpha.max(sc), self.beta)
   }

   fn score_to_range(self, sc: Score) -> (Score, Score) {

      let min = if sc > self.alpha { sc } else { -Score::Inf };
      let max = if sc < self.beta  { sc } else {  Score::Inf };

      debug_assert!((min, max) != (-Score::Inf, Score::Inf));

      (min, max)
   }
}

impl ops::Neg for AB {

   type Output = Self;

   fn neg(self) -> Self::Output {
      Self::new(-self.beta, -self.alpha)
   }
}

impl Bound {

   fn new(sc: Score, ab: AB) -> Self {

      if sc <= ab.alpha {
         Self::Upper
      } else if sc >= ab.beta {
         Self::Lower
      } else {
         debug_assert!(sc > ab.alpha && sc < ab.beta);
         Self::Exact
      }
   }

   pub fn is_lower(self) -> bool {
      self != Self::Upper
   }

   pub fn is_upper(self) -> bool {
      self != Self::Lower
   }

   fn to_range(self, sc: Score) -> (Score, Score) {

      let min = if self.is_lower() { sc } else { -Score::Inf };
      let max = if self.is_upper() { sc } else {  Score::Inf };

      (min, max)
   }
}

impl Move_Info {

   fn new(mv: Move, bd: &Board, in_check: bool, stage: Stage) -> Self {

      Self {

         move_: mv,
         stage,

         is_cap:   mv.is_cap (bd),
         is_prom:  mv.is_prom(bd),
         is_check: None,
         is_recap: None,
         is_safe:  None,
      }
   }

   fn is_cap(&mut self) -> bool {
      self.is_cap
   }

   fn is_prom(&mut self) -> bool {
      self.is_prom
   }

   fn is_check(&mut self, bd: &Board) -> bool {

      if let Some(res) = self.is_check { return res }

      let res = self.move_.is_check(bd);
      self.is_check = Some(res);
      res
   }

   fn is_recap(&mut self, bd: &Board) -> bool {

      if let Some(res) = self.is_recap { return res }

      let res = self.is_safe(bd) && self.move_.is_recap(bd);
      self.is_recap = Some(res);
      res
   }

   fn is_safe(&mut self, bd: &Board) -> bool {

      if let Some(res) = self.is_safe { return res }

      if self.stage == Stage::Bad { debug_assert!(!self.move_.is_safe(bd) || self.move_.is_prom(bd)) };

      let res = self.stage != Stage::Bad && self.move_.is_safe(bd);
      self.is_safe = Some(res);
      res
   }
}

