
// modules

use super::{Score};

use crate::prelude::*;
use crate::game::attack;

// types

pub struct Pos<'b> {

   board: &'b Board,
   turn:  Side,

   blocker: BB,
   to: Square,

   mat: Score,
}

// functions

pub fn move_is_safe(mv: Move, bd: &Board) -> bool {

   let pc = mv.piece(bd);

   if pc == Piece::King { return true }

   if let Some(cp) = mv.cap(bd) && mat(cp) >= mat(pc) {
      return true
   }

   see_move(mv, bd) >= Score(0)
}

pub fn move_is_win(mv: Move, bd: &Board) -> bool {

   debug_assert!(mv.is_cap(bd));

   let pc = mv.piece(bd);

   if pc == Piece::King { return true }

   if let Some(cp) = mv.cap(bd) && mat(cp) > mat(pc) {
      return true
   }

   see_move(mv, bd) > Score(0)
}

pub fn max(mv: Move, bd: &Board) -> Score {

   let mut res = Score(0);

   if let Some(cp) = mv.cap(bd) { res += mat(cp) + Score(100) };

   if let Some(pm) = mv.auto_queen(bd) {
      res += prom_val(pm) + Score(50);
   } else if mv.is_push(bd) || mv.is_castle(bd) {
      res += Score(50);
   } else { // HACK for evasion
      res += Score(50);
   }

   res
}

pub fn see_move(mv: Move, bd: &Board) -> Score {

   debug_assert!(!mv.is_castle(bd));

   let pos = &mut Pos::new(bd, mv.to());

   let mut sc = pos.do_move(mv.from());
   if let Some(pm) = mv.auto_queen(bd) { sc += pos.do_prom(pm) }

   sc - search(pos)
}

fn search(pos: &mut Pos<'_>) -> Score {

   if let Some(from) = pos.get_lva() {
      let sc = pos.do_move(from);
      (sc - search(pos)).max(Score(0))
   } else {
      Score(0)
   }
}

impl<'b> Pos<'b> {

   fn new(bd: &'b Board, to: Square) -> Self {

      let turn = bd.turn();

      let blocker = bd.all();

      let mat = if let Some((pc, sd)) = bd.square(to) {
         debug_assert!(sd == turn.opp());
         mat(pc)
      } else {
         Score(0)
      };

      Self {

         board: bd,
         turn,

         blocker,
         to,

         mat,
      }
   }

#[must_use]
   fn do_move(&mut self, from: Square) -> Score {

      let (pc, sd) = self.board.square(from).unwrap();
      debug_assert!(sd == self.turn);

      self.blocker.clear(from);

      let mat  = self.mat;
      self.mat = self::mat(pc);

      self.turn = self.turn.opp();

      mat
   }

#[must_use]
   fn do_prom(&mut self, pm: Piece) -> Score {

      debug_assert!(self.mat == mat(Piece::Pawn));
      self.mat = mat(pm);

      prom_val(pm)
   }

   fn get_lva(&self) -> Option<Square> {

      let sd = self.turn;

      let table = &self.board.global.table_bb;

      for pc in Piece::iter() {

         let froms = self.board.piece(pc, sd)
                   & self.blocker
                   & table.attacks_to(pc, sd, self.to);

         for from in froms {
            if table.line_is_empty(self.to, from, self.blocker) && self.can_capture(from) {
               return Some(from);
            }
         }
      }

      None
   }

   fn can_capture(&self, from: Square) -> bool {

      let (pc, sd) = self.board.square(from).unwrap();

      let bd   = self.board;
      let king = bd.king(sd);

      if attack::is_pinned_by_aux(king, from, sd.opp(), bd, self.blocker) {
         bd.global.table_bb.pin_tos(king, from).has(self.to)
      } else {
         true
      }
   }
}

fn prom_val(pc: Piece) -> Score {
   Score(mat(pc).val() - mat(Piece::Pawn).val() * 2)
}

fn mat(pc: Piece) -> Score {
   let piece_mat = [100, 400, 400, 600, 1_200, 3_600];
   Score(piece_mat[pc.index()])
}

