
// modules

use crate::prelude::*;

use super::{pawn, Colour};
use super::square::{self, Inc, Sq};

use crate::util::bit;

use std::ops;

// constants

const Vec_Knight: [Vec; 8] = [
   (-1, -2), ( 1, -2),
   (-2, -1), ( 2, -1),
   (-2,  1), ( 2,  1),
   (-1,  2), ( 1,  2),
];

const Vec_Bishop: [Vec; 4] = [
   (-1, -1), ( 1, -1),
   (-1,  1), ( 1,  1),
];

const Vec_Rook: [Vec; 4] = [
   ( 0, -1), (-1,  0),
   ( 1,  0), ( 0,  1),
];

const Vec_Queen: [Vec; 8] = [
   (-1, -1), ( 0, -1), ( 1, -1),
   (-1,  0),           ( 1,  0),
   (-1,  1), ( 0,  1), ( 1,  1),
];

const Vec_King: [Vec; 8] = [
   (-1, -1), ( 0, -1), ( 1, -1),
   (-1,  0),           ( 1,  0),
   (-1,  1), ( 0,  1), ( 1,  1),
];

// types

#[derive(Clone, Copy, PartialEq, Eq, Default)]
pub struct BB(Bits);

pub struct Iter_BB(BB);

pub struct Iter_Ray {
   square: Square,
   vector: (Inc, Inc),
}

pub struct Table_BB {

   file:  [BB; Square::Size as usize],
   rank:  [BB; Square::Size as usize],
   files: [BB; Square::Size as usize],

   front: [[BB; Square::Size as usize]; Side::Size as usize],

   pawn_moves: [[BB; Square::Size as usize]; Side::Size as usize],
   pawn_caps:  [[BB; Square::Size as usize]; Side::Size as usize],

   moves:   [[BB; Square::Size as usize]; Piece ::Size as usize],
   blocker: [[BB; Square::Size as usize]; Piece ::Size as usize],
   zone:    [[BB; Piece ::Size as usize]; Square::Size as usize],

   between: [[BB; Square::Size as usize]; Square::Size as usize],
   beyond:  [[BB; Square::Size as usize]; Square::Size as usize],
}

type Vec = (Inc, Inc);

type Bit = Sq;
pub type Bits = u64;

// functions

impl BB {

   const Squares: Bits = !0; // assumes dense square numbering

   pub const fn empty() -> Self {
      Self(0)
   }

   pub const fn full() -> Self {
      Self(Self::Squares)
   }

   pub const fn square(sq: Square) -> Self {
      Self(Self::bit(sq))
   }

   pub fn left(fl: Inc) -> Self {
      debug_assert!(fl >= 0 && fl < Square::File_Size);
      (0 .. fl).map(Self::file).fold(Self::empty(), |s, x| s | x)
   }

   pub fn right(fl: Inc) -> Self {
      debug_assert!(fl >= 0 && fl < Square::File_Size);
      ((fl + 1) .. Square::File_Size).map(Self::file).fold(Self::empty(), |s, x| s | x)
   }

   pub fn down(rk: Inc) -> Self {
      debug_assert!(rk >= 0 && rk < Square::Rank_Size);
      (0 .. rk).map(Self::rank).fold(Self::empty(), |s, x| s | x)
   }

   pub fn up(rk: Inc) -> Self {
      debug_assert!(rk >= 0 && rk < Square::Rank_Size);
      ((rk + 1) .. Square::Rank_Size).map(Self::rank).fold(Self::empty(), |s, x| s | x)
   }

   pub fn front(rk: Inc, sd: Side) -> Self {

      match sd {
         Side::White => Self::up  (rk),
         Side::Black => Self::down(rk),
      }
   }

   pub fn files(fl: Inc) -> Self {

      debug_assert!(fl >= 0 && fl < Square::File_Size);

      let mut res = Self::empty();
      if fl != 0 { res |= Self::file(fl - 1) }
      if fl != 7 { res |= Self::file(fl + 1) }

      res
   }

   pub const fn file(fl: Inc) -> Self {
      debug_assert!(fl >= 0 && fl < Square::File_Size);
      Self(bit::bits(Square::Rank_Size as Sq) << (fl * Square::Rank_Size))
   }

   pub const fn rank(rk: Inc) -> Self {
      debug_assert!(rk >= 0 && rk < Square::Rank_Size);
      Self(0x0101010101010101 << rk)
   }

   pub const fn rank_side(rk: Inc, sd: Side) -> Self {
      Self::rank(square::rank_side(rk, sd))
   }

   pub fn pawns() -> Self {

      Self::full()
    - Self::rank(0)
    - Self::rank(Square::Rank_Size - 1)
   }

   pub fn pawn(sd: Side) -> Self {
      Self::full() - Self::prom(sd)
   }

   pub const fn same_colour(sq: Square) -> Self {
      Self::colour(sq.colour())
   }

   pub const fn colour(cl: Colour) -> Self {

      match cl {
         Colour::Dark  => Self(0xAA55AA55AA55AA55),
         Colour::Light => Self(0x55AA55AA55AA55AA),
      }
   }

   pub const fn prom(sd: Side) -> Self {
      Self::rank_side(0, sd.opp())
   }

   pub const fn seventh(sd: Side) -> Self {
      Self::rank_side(1, sd.opp())
   }

   pub const fn is_empty(self) -> bool {
      self.0 == 0
   }

   pub const fn has(self, sq: Square) -> bool {
      (self.0 & Self::bit(sq)) != 0
   }

   pub const fn set(&mut self, sq: Square) {
      self.0 |= Self::bit(sq);
   }

   pub const fn clear(&mut self, sq: Square) {
      self.0 &= !Self::bit(sq);
   }

   pub const fn shift(self, inc: Inc) -> Bits {
      bit::shift(self.0, inc)
   }

   pub const fn is_single(self) -> bool {
      bit::is_single(self.0)
   }

   pub const fn is_subset(self, r: Self) -> bool {
      bit::is_subset(self.0, r.0)
   }

   pub const fn is_disjoint(self, r: Self) -> bool {
      bit::is_disjoint(self.0, r.0)
   }

   pub const fn first(self) -> Square {
      debug_assert!(!self.is_empty());
      Square::from_int(bit::first(self.0))
   }

   pub const fn rest(self) -> Self {
      debug_assert!(!self.is_empty());
      Self(bit::rest(self.0))
   }

   fn remove_first(&mut self) -> Square { // TODO: inline?
      let sq = self.first();
      *self  = self.rest();
      sq
   }

   pub const fn count(self) -> Bit {
      bit::count(self.0)
   }

   pub fn iter(self) -> Iter_BB {
      Iter_BB::new(self)
   }

   const fn bit(sq: Square) -> Bits {
      1 << sq.val()
   }
}

impl From<Bits> for BB {

   fn from(bits: Bits) -> Self {
      Self::full() & bits
   }
}

impl ops::BitOr for BB {

   type Output = Self;

   fn bitor(self, rhs: Self) -> Self {
      Self(self.0 | rhs.0)
   }
}

impl ops::BitOr<Square> for BB {

   type Output = Self;

   fn bitor(self, rhs: Square) -> Self {
      self | Self::square(rhs)
   }
}

impl ops::BitOrAssign for BB {

   fn bitor_assign(&mut self, rhs: Self) {
      *self = *self | rhs;
   }
}

impl ops::Sub for BB {

   type Output = Self;

   fn sub(self, rhs: Self) -> Self {
      Self(self.0 & !rhs.0)
   }
}

impl ops::Sub<Square> for BB {

   type Output = Self;

   fn sub(self, rhs: Square) -> Self {
      self - Self::square(rhs)
   }
}

impl ops::Sub<Bits> for BB {

   type Output = Self;

   fn sub(self, rhs: Bits) -> Self {
      Self(self.0 & !rhs)
   }
}

impl ops::SubAssign for BB {

   fn sub_assign(&mut self, rhs: Self) {
      *self = *self - rhs;
   }
}

impl ops::SubAssign<Bits> for BB {

   fn sub_assign(&mut self, rhs: Bits) {
      *self = *self - rhs;
   }
}

impl ops::BitAnd for BB {

   type Output = Self;

   fn bitand(self, rhs: Self) -> Self {
      Self(self.0 & rhs.0)
   }
}

impl ops::BitAnd<Bits> for BB {

   type Output = Self;

   fn bitand(self, rhs: Bits) -> Self {
      Self(self.0 & rhs)
   }
}

impl ops::BitAndAssign for BB {

   fn bitand_assign(&mut self, rhs: Self) {
      *self = *self & rhs;
   }
}

impl FromIterator<Square> for BB {

   fn from_iter<I: IntoIterator<Item = Square>>(iter: I) -> Self {

      let mut res = Self::default();
      res.extend(iter);
      res
   }
}

impl Extend<Square> for BB {

   fn extend<I: IntoIterator<Item = Square>>(&mut self, iter: I) {

      for sq in iter {
         self.set(sq);
      }
   }
}

impl IntoIterator for BB {

   type Item = Square;
   type IntoIter = Iter_BB;

   fn into_iter(self) -> Self::IntoIter {
      Iter_BB::new(self)
   }
}

impl Iter_BB {

   fn new(bb: BB) -> Self {
      Self(bb)
   }
}

impl Iterator for Iter_BB {

   type Item = Square;

   fn next(&mut self) -> Option<Self::Item> {

      if self.0.is_empty() {
         None
      } else {
         Some(self.0.remove_first())
      }
   }

   fn size_hint(&self) -> (usize, Option<usize>) {
      let size = self.0.count() as usize;
      (size, Some(size))
   }
}

impl ExactSizeIterator for Iter_BB {

   fn len(&self) -> usize {
      self.0.count().into()
   }
}

impl Table_BB {

   pub fn new() -> Self {

      use Piece::*;

      let mut file  = [BB::empty(); Square::Size as usize];
      let mut rank  = [BB::empty(); Square::Size as usize];
      let mut files = [BB::empty(); Square::Size as usize];

      let mut front = [[BB::empty(); Square::Size as usize]; Side::Size as usize];

      let mut pawn_moves = [[BB::empty(); Square::Size as usize]; Side::Size as usize];
      let mut pawn_caps  = [[BB::empty(); Square::Size as usize]; Side::Size as usize];

      let mut moves   = [[BB::empty(); Square::Size as usize]; Piece ::Size as usize];
      let mut blocker = [[BB::empty(); Square::Size as usize]; Piece ::Size as usize];
      let mut zone    = [[BB::empty(); Piece ::Size as usize]; Square::Size as usize];

      let mut between = [[BB::empty(); Square::Size as usize]; Square::Size as usize];
      let mut beyond  = [[BB::empty(); Square::Size as usize]; Square::Size as usize];

      // squares

      for sq in Square::iter() {

         let (fl, rk) = sq.coords();

         file [sq.index()] = BB::file (fl);
         rank [sq.index()] = BB::rank (rk);
         files[sq.index()] = BB::files(fl);

         for sd in Side::iter() {
            front[sd.index()][sq.index()] = BB::front(rk, sd);
         }
      }

      // pawns

      for sd in Side::iter() {
         for from in BB::pawn(sd).iter() {
            pawn_moves[sd.index()][from.index()] = BB::from(pawn::moves_from(sd, from));
            pawn_caps [sd.index()][from.index()] = BB::from(pawn:: caps_from(sd, from));
         }
      }

      // moves/blocker

      for from in Square::iter() {

         moves[Knight.index()][from.index()] = leaper(from, &Vec_Knight);
         moves[Bishop.index()][from.index()] = slider(from, &Vec_Bishop);
         moves[Rook  .index()][from.index()] = slider(from, &Vec_Rook);
         moves[Queen .index()][from.index()] = slider(from, &Vec_Queen);
         moves[King  .index()][from.index()] = leaper(from, &Vec_King);

         blocker[Bishop.index()][from.index()] = self::blocker(from, &Vec_Bishop);
         blocker[Rook  .index()][from.index()] = self::blocker(from, &Vec_Rook);
         blocker[Queen .index()][from.index()] = self::blocker(from, &Vec_Queen);
      }

      // zone

      for king in Square::iter() {
         for to in moves[King.index()][king.index()] { // zone
            for pc in Piece::iter() {
               zone[king.index()][pc.index()] |= moves[pc.index()][to.index()];
            }
         }
      }

      // between/beyond

      for from in Square::iter() {

         for vec in Vec_Queen {

            let mut bb = BB::empty();

            for to in ray_iter(from, vec) {

               between[from.index()][to.index()] = bb;
               beyond [from.index()][to.index()] = ray_bb(to, vec);

               bb.set(to);
            }
         }
      }

      Self {

         file,
         rank,
         files,

         front,

         pawn_moves,
         pawn_caps,

         moves,
         blocker,
         zone,

         between,
         beyond,
      }
   }

   pub const fn file(&self, sq: Square) -> BB {
      self.file[sq.index()]
   }

   pub const fn files(&self, sq: Square) -> BB {
      self.files[sq.index()]
   }

   pub const fn rank(&self, sq: Square) -> BB {
      self.rank[sq.index()]
   }

   pub const fn front(&self, sq: Square, sd: Side) -> BB {
      self.front[sd.index()][sq.index()]
   }

   pub const fn pawn_moves_from(&self, sd: Side, from: Square) -> BB {
      self.pawn_moves[sd.index()][from.index()]
   }

   pub const fn pawn_caps_from(&self, sd: Side, from: Square) -> BB {
      self.pawn_caps[sd.index()][from.index()]
   }

   pub const fn pawn_caps_to(&self, sd: Side, to: Square) -> BB {
      self.pawn_caps[sd.opp().index()][to.index()]
   }

   pub fn moves(&self, pc: Piece, from: Square) -> BB {
      debug_assert!(pc != Piece::Pawn);
      self.moves[pc.index()][from.index()]
   }

   pub fn zone(&self, king: Square, pc: Piece) -> BB {
      debug_assert!(pc != Piece::Pawn);
      self.zone[king.index()][pc.index()]
   }

   pub fn attacks_to(&self, pc: Piece, sd: Side, to: Square) -> BB {

      if pc == Piece::Pawn {
         self.pawn_caps_to(sd, to)
      } else {
         self.moves(pc, to)
      }
   }

   pub const fn blocker(&self, pc: Piece, from: Square) -> BB {
      self.blocker[pc.index()][from.index()]
   }

   pub const fn between(&self, from: Square, to: Square) -> BB {
      self.between[from.index()][to.index()]
   }

   pub fn until(&self, from: Square, to: Square) -> BB {
      self.between(from, to) | to
   }

   pub const fn beyond(&self, from: Square, to: Square) -> BB {
      self.beyond[from.index()][to.index()]
   }

   pub fn pin_tos(&self, king: Square, from: Square) -> BB {
      self.between(king, from)
    | self.beyond (king, from)
   }

   pub fn attack(&self, pc: Piece, from: Square, blocker: BB) -> BB {

      let mut tos = self.moves(pc, from);

      for sq in blocker & self.blocker(pc, from) {
         tos -= self.beyond(from, sq);
      }

      tos
   }

   pub fn line_is_empty(&self, from: Square, to: Square, blocker: BB) -> bool {
      BB::is_disjoint(blocker, self.between(from, to))
   }
}

impl Iter_Ray {

   fn new(square: Square, vector: Vec) -> Self {
      Self { square, vector }
   }
}

impl Iterator for Iter_Ray {

   type Item = Square;

   fn next(&mut self) -> Option<Self::Item> {
      self.square = next(self.square, self.vector)?;
      Some(self.square)
   }
}

fn leaper(from: Square, vecs: &[Vec]) -> BB {

   vecs.iter()
       .flat_map(|&vec| next(from, vec))
       .collect()
}

fn slider(from: Square, vecs: &[Vec]) -> BB {

   vecs.iter()
       .flat_map(|&vec| ray_iter(from, vec))
       .collect()
}

fn blocker(from: Square, vecs: &[Vec]) -> BB {

   let mut res = BB::empty();

   for &vec in vecs {

      let mut sq = from;

      while let Some(next) = self::next(sq, vec) {
         if self::next(next, vec).is_none() { break }
         res.set(next);
         sq = next;
      }
   }

   res
}

fn ray_bb(from: Square, vec: Vec) -> BB {
   ray_iter(from, vec).collect()
}

fn ray_iter(from: Square, vec: Vec) -> impl Iterator<Item = Square> {
   Iter_Ray::new(from, vec)
}

fn next(from: Square, vec: Vec) -> Option<Square> {
   from + vec
}

