
// modules

use crate::prelude::*;

use super::{attack, legal, pawn};

use crate::util::prelude::*;

// functions

pub fn gen_caps(list: &mut impl Grow<Item = Move>, bd: &Board) {

   let sd = bd.turn();

   list.clear();

   let mut tos = bd.side(sd.opp());
   add_moves_slow(list, bd, BB::full(), tos);

   if let Some(ep) = bd.ep_square() {
      tos.set(ep);
   }

   add_pawn_caps(list, bd, BB::full(), tos);
}

pub fn add_proms(list: &mut impl Grow<Item = Move>, bd: &Board) {
   let sd = bd.turn();
   list.append(pawn::moves_froms(bd.pawn(sd), BB::prom(sd) & bd.empty(), pawn::inc_move(sd)));
}

pub fn add_push(list: &mut impl Grow<Item = Move>, bd: &Board) {
   let sd = bd.turn();
   list.append(pawn::moves_froms(bd.pawn(sd), BB::seventh(sd) & bd.empty(), pawn::inc_move(sd)));
}

pub fn add_quiets(list: &mut impl Grow<Item = Move>, bd: &Board) {

   let sd = bd.turn();

   add_castle     (list, bd);
   add_moves      (list, bd, BB::full(), bd.empty());
   add_pawn_quiets(list, bd, BB::full(), bd.empty());
}

pub fn add_safe(list: &mut impl Grow<Item = Move>, bd: &Board) {

   let sd  = bd.turn();
   let tos = bd.empty() - pawn::caps_froms(bd, sd.opp());

   add_castle     (list, bd);
   add_moves      (list, bd, BB::full(), tos);
   add_pawn_quiets(list, bd, BB::full(), bd.empty());
}

pub fn gen_eva_caps(list: &mut impl Grow<Item = Move>, bd: &Board, checks: BB) {

   debug_assert!(!checks.is_empty());

   list.clear();

   let sd   = bd.turn();
   let king = bd.king(sd);

   if checks.is_single() { add_caps_to(list, bd, BB::full() - king, checks.first()) }
   add_king_moves(list, bd, king, bd.side(sd.opp()));
}

pub fn add_eva_quiets(list: &mut impl Grow<Item = Move>, bd: &Board, checks: BB) {

   debug_assert!(!checks.is_empty());

   let sd   = bd.turn();
   let king = bd.king(sd);
   let tos  = bd.empty();

   add_king_moves(list, bd, king, tos);

   if checks.is_single() {

      let line = bd.global.table_bb.between(king, checks.first());

      add_moves_slow (list, bd, BB::full() - king, tos & line);
      add_pawn_quiets(list, bd, BB::full(),        tos & line);
   }
}

pub fn gen_recaps(list: &mut impl Grow<Item = Move>, bd: &Board) {
   list.clear();
   if let Some(to) = bd.last_cap() { add_caps_to(list, bd, BB::full(), to) }
}

pub fn add_checks_fast(list: &mut impl Grow<Item = Move>, bd: &Board) {

   let sd   = bd.turn();
   let king = bd.king(sd.opp());

   let table = &bd.global.table_bb;

   // direct checks

   let tos = bd.empty() - pawn::caps_froms(bd, sd.opp()); // safe

   for p in 1 .. 5 { // skip pawn and king

      let pc = Piece::from_int(p);

      for from in bd.piece(pc, sd) {
         for to in tos & table.moves(pc, from) & table.moves(pc, king) {
            if bd.line_is_empty(from, to) && bd.line_is_empty(king, to) {
               list.add(Move::new(from, to));
            }
         }
      }
   }

   // ignore pawns
}

fn add_moves(list: &mut impl Grow<Item = Move>, bd: &Board, froms: BB, tos: BB) {

   let sd = bd.turn();

   let table = &bd.global.table_bb;

   // knight

   for from in froms & bd.piece(Piece::Knight, sd) {

      let pc = Piece::Knight;

      for to in tos & table.moves(pc, from) {
         list.add(Move::new(from, to));
      }
   }

   // bishop

   for from in froms & bd.bishop_queen(sd) {

      let pc = Piece::Bishop;

      for to in tos & bd.global.attack_bishop(from, bd.all()) {
         debug_assert!(bd.line_is_empty(from, to));
         list.add(Move::new(from, to));
      }
   }

   // rook

   for from in froms & bd.rook_queen(sd) {

      let pc = Piece::Rook;

      for to in tos & bd.global.attack_rook(from, bd.all()) {
         debug_assert!(bd.line_is_empty(from, to));
         list.add(Move::new(from, to));
      }
   }

   // king

   {
      let pc   = Piece::King;
      let from = bd.king(sd);

      if froms.has(from) {
         for to in tos & table.moves(pc, from) {
            list.add(Move::new(from, to));
         }
      }
   }

   // skip pawns
}

fn add_moves_slow(list: &mut impl Grow<Item = Move>, bd: &Board, froms: BB, tos: BB) {

   let sd = bd.turn();

   let table = &bd.global.table_bb;

   for pc in Piece::no_pawn() {
      for from in froms & bd.piece(pc, sd) {
         for to in tos & table.moves(pc, from) {
            if bd.line_is_empty(from, to) {
               list.add(Move::new(from, to));
            }
         }
      }
   }

   // skip pawns
}

pub fn add_pawn_moves(list: &mut impl Grow<Item = Move>, bd: &Board, froms: BB, tos: BB) {
   add_pawn_caps  (list, bd, froms, tos);
   add_pawn_quiets(list, bd, froms, tos);
}

fn add_pawn_caps(list: &mut impl Grow<Item = Move>, bd: &Board, froms: BB, tos: BB) {

   let sd = bd.turn();

   let mut ts = bd.side(sd.opp());
   if let Some(ep) = bd.ep_square() { ts.set(ep) }

   list.append(pawn::moves_froms(froms & bd.pawn(sd), tos & ts, pawn::inc_cap_left (sd)));
   list.append(pawn::moves_froms(froms & bd.pawn(sd), tos & ts, pawn::inc_cap_right(sd)));
}

fn add_pawn_quiets(list: &mut impl Grow<Item = Move>, bd: &Board, froms: BB, tos: BB) {

   let sd = bd.turn();

   let fs  = froms & bd.pawn(sd);
   let ts  = tos   & bd.empty();
   let inc = pawn::inc_move(sd);

   list.append(pawn::moves_froms(fs, ts, inc));
   list.append(pawn::moves_froms(fs & BB::rank_side(1, sd) & bd.empty().shift(-inc), ts, inc * 2));
}

pub fn add_king_moves(list: &mut impl Grow<Item = Move>, bd: &Board, from: Square, tos: BB) {

   debug_assert!(bd.king(bd.turn()) == from);

   for to in tos & bd.global.table_bb.moves(Piece::King, from) {
      list.add(Move::new(from, to));
   }
}

pub fn add_castle(list: &mut impl Grow<Item = Move>, bd: &Board) {

   let sd   = bd.turn();
   let king = bd.king(sd);

   for rook in bd.castle_side(sd) {
      if legal::castle_is_legal(king, rook, bd) {
         list.add(Move::new(king, rook));
      }
   }
}

pub fn add_moves_from(list: &mut impl Grow<Item = Move>, bd: &Board, pc: Piece, from: Square, tos: BB) {

   let sd = bd.turn();

   debug_assert!(bd.square(from) == Some((pc, sd)));

   for to in tos & attack::piece_tos(pc, sd, from, bd) {
      if bd.line_is_empty(from, to) {
         list.add(Move::new(from, to));
      }
   }
}

fn add_caps_to(list: &mut impl Grow<Item = Move>, bd: &Board, froms: BB, to: Square) {

   let sd = bd.turn();

   debug_assert!(bd.square_is_side(to, sd.opp()));

   if let Some(ep) = bd.ep_square()
   && to == ep.ep_opp()
   {
      let pc = Piece::Pawn;
      let to = ep;

      for from in froms & bd.piece(pc, sd) & bd.global.table_bb.attacks_to(pc, sd, to) {
         list.add(Move::new(from, to));
      }
   }

   for pc in Piece::iter() {
      for from in froms & bd.piece(pc, sd) & bd.global.table_bb.attacks_to(pc, sd, to) {
         if bd.line_is_empty(from, to) {
            list.add(Move::new(from, to));
         }
      }
   }
}

pub fn count_caps_to(bd: &Board, froms: BB, to: Square) -> u16 {

   let sd = bd.turn();

   debug_assert!(bd.square_is_side(to, sd.opp()));

   let mut res = 0;

   for pc in Piece::iter() {
      for from in froms & bd.piece(pc, sd) & bd.global.table_bb.attacks_to(pc, sd, to) {
         if bd.line_is_empty(from, to) {
            res += 1;
         }
      }
   }

   res
}

