
// modules

use crate::prelude::*;

use super::{attack, gen_, pawn, Wing};

use crate::engine::{Move_Long}; // HACK

use crate::util::prelude::*;

// types

pub struct Legal_Info {

   king: Square,

   checks:    BB,
   piece_tos: BB,
   pawn_tos:  BB,
   pins:      BB,
}

// functions

pub fn has_legal(bd: &Board) -> bool {
   !legals(bd).is_empty()
}

pub fn legals(bd: &Board) -> Vec<Move> {

   let li = Legal_Info::new(bd);

   let mut list = Vec::with_capacity(64);
   li.gen_legals(&mut list, bd);
   list
}

impl Legal_Info {

   pub fn new(bd: &Board) -> Self {

      let sd = bd.turn();
      let xd = sd.opp();

      let table = &bd.global.table_bb;

      let king = bd.king(sd);

      let mut checks    = BB::empty();
      let mut piece_tos = BB::full () - bd.side(sd);
      let mut pins      = BB::empty();

      // leapers

      checks |= bd.pawn(xd) & table.pawn_caps_to(xd, king);
      checks |= bd.piece(Piece::Knight, xd) & table.moves(Piece::Knight, king);

      debug_assert!(checks.count() <= 1);
      if !checks.is_empty() { piece_tos &= checks }

      // sliders

      for from in attack::pseudo_sliders_to(bd, king, xd) {

         let between = table.between(king, from);
         let blocker = bd.all() & between;

         if blocker.is_empty() { // direct check
            checks.set(from);
            piece_tos &= between | from;
         } else if blocker.is_single() { // pinned piece (any side)
            pins |= blocker;
         }
      }

      let mut pawn_tos = piece_tos;

      if let Some(ep) = bd.ep_square()
      && !(checks & bd.pawn(xd)).is_empty()
      {
         pawn_tos.set(ep);
      }

      Self { king, checks, piece_tos, pawn_tos, pins }
   }

   pub fn long_is_pseudo(&self, long: Move_Long, bd: &Board) -> bool {

      let mv: Move = long.into();

      self.is_pseudo(mv, bd)
      && mv.piece(bd) == long.piece()
      && mv.cap  (bd) == long.cap()
   }

   pub fn is_pseudo(&self, mv: Move, bd: &Board) -> bool {

      let from = mv.from();
      let to   = mv.to();

      let Some((pc, sd)) = bd.square(from) else { return false };

      sd == bd.turn()
      && self.piece_can(pc, sd, from, to, bd)
      && bd.line_is_empty(from, to)
   }

   fn piece_can(&self, pc: Piece, sd: Side, from: Square, to: Square, bd: &Board) -> bool {

      if pc == Piece::King && bd.castle_side(sd).has(to) {

         castle_is_legal(from, to, bd)

      } else {

         attack::piece_tos(pc, sd, from, bd).has(to)
         && !bd.square_is_side(to, sd)
         && (pc == Piece::King || self.piece_tos.has(to) || (pc == Piece::Pawn && self.pawn_tos.has(to)))
      }
   }

   pub fn is_legal(&self, mv: Move, bd: &Board) -> bool {

      let sd = bd.turn();

      let from = mv.from();
      let to   = mv.to();

      let table = &bd.global.table_bb;

      if from == self.king {

         !attack::is_attacked_through_king(bd, mv.king_to(bd), sd.opp())

      } else if self.pins.has(from) {

         table.pin_tos(self.king, from).has(to)

      } else if mv.is_ep(bd) && from.rank() == self.king.rank() {

         for rook in bd.rook_queen(sd.opp()) & BB::rank(self.king.rank()) {
            if table.line_is_empty(self.king, rook, bd.all() - from - to.ep_opp()) {
               return false;
            }
         }

         true

      } else {

         true
      }
   }

   pub fn seem_legal(&self, bd: &Board) -> bool { // probably has a legal move (can rarely be wrong, both ways)

      assert!(!self.in_check());

      let sd = bd.turn();

      let table = &bd.global.table_bb;

      // non-pinned piece

      let froms = bd.pieces(sd) - self.pins;
      if !froms.is_empty() { return true } // HACK: assume one piece have a legal move

      // non-pinned pawn move

      let froms = bd.pawn(sd) - self.pins;

      let tos = self.pawn_tos & bd.empty() & froms.shift(pawn::inc_move(sd));
      if !tos.is_empty() { return true }

      // king

      let tos = table.moves(Piece::King, self.king) & comp_zone(bd, sd);
      if !tos.is_empty() { return true }

      false
   }

   pub fn gen_legals(&self, list: &mut impl Grow<Item = Move>, bd: &Board) {

      list.clear();

      let sd = bd.turn();

      let table = &bd.global.table_bb;

      let king_tos = comp_zone(bd, sd);

      // knight

      for from in bd.piece(Piece::Knight, sd) - self.pins {
         for to in self.piece_tos & table.moves(Piece::Knight, from) {
            list.add(Move::new(from, to));
         }
      }

      // bishop

      for from in bd.bishop_queen(sd) - self.pins {

         for to in self.piece_tos & bd.global.attack_bishop(from, bd.all()) {
            debug_assert!(bd.line_is_empty(from, to));
            list.add(Move::new(from, to));
         }
      }

      // rook

      for from in bd.rook_queen(sd) - self.pins {

         for to in self.piece_tos & bd.global.attack_rook(from, bd.all()) {
            debug_assert!(bd.line_is_empty(from, to));
            list.add(Move::new(from, to));
         }
      }

      // others

      if !self.in_check() {

         let king = self.king;

         for rook in bd.castle_side(sd) {
            if castle_is_legal(king, rook, bd) {
               list.add(Move::new(king, rook));
            }
         }
      }

      gen_::add_king_moves(list, bd, self.king, king_tos);
      gen_::add_pawn_moves(list, bd, bd.pawn(sd) - self.pins, self.pawn_tos);

      // pinned bishop

      let pc = Piece::Bishop;

      for from in self.pins & bd.bishop_queen(sd) & table.moves(pc, self.king) {

         for to in self.piece_tos & table.moves(pc, self.king) & table.moves(pc, from) {
            if bd.line_is_empty(from, to) {
               list.add(Move::new(from, to));
            }
         }
      }

      // pinned rook

      let pc = Piece::Rook;

      for from in self.pins & bd.rook_queen(sd) & table.moves(pc, self.king) {

         for to in self.piece_tos & table.moves(pc, self.king) & table.moves(pc, from) {
            if bd.line_is_empty(from, to) {
               list.add(Move::new(from, to));
            }
         }
      }

      // pinned pawns

      for from in self.pins & bd.pawn(sd) {
         let tos = self.pawn_tos & table.pin_tos(self.king, from);
         gen_::add_moves_from(list, bd, Piece::Pawn, from, tos);
      }
   }

   pub fn in_check(&self) -> bool {
      !self.checks.is_empty()
   }

   pub fn checks(&self) -> BB {
      self.checks
   }
}

fn comp_zone(bd: &Board, sd: Side) -> BB {

   let xd   = sd.opp();
   let king = bd.king(sd);

   let table = &bd.global.table_bb;

   let mut res = table.moves(Piece::King, king) - bd.side(sd);

   // pawns

   res = res - pawn::caps_froms(bd, xd);

   // pieces

   let blocker = bd.all() - king;

   for pc in Piece::no_pawn() {
      for from in bd.piece(pc, xd) & table.zone(king, pc) {
         for to in res & table.moves(pc, from) {
            if table.line_is_empty(from, to, blocker) { res.clear(to) }
         }
      }
   }

   res
}

pub fn castle_is_legal(kf: Square, rf: Square, bd: &Board) -> bool {

   let sd = bd.turn();

   debug_assert!(bd.square(rf) == Some((Piece::Rook, sd)));

   let table = &bd.global.table_bb;

   if bd.line_is_empty(kf, rf) {

      let wing     = Wing::new(kf, rf);
      let (kt, rt) = bd.global.table_rule.king_rook_to(wing, sd);
      let king_tos = table.until(kf, kt) | kf; // TODO: remove kt? #
      let blocker  = bd.all() - kf - rf;

      if BB::is_disjoint(blocker, table.until(kf, kt))
      && BB::is_disjoint(blocker, table.until(rf, rt))
      && !attack::is_attacked_tos(bd, king_tos, sd.opp(), blocker)
      {
         return true;
      }
   }

   false
}

