
// modules

use crate::prelude::*;

use crate::util;

use std::{fmt, mem, ops, str};

// types

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct Square(Sq);

#[derive(Clone, Copy, PartialEq, Eq)]
pub enum Colour { Dark, Light }

pub type Sq  = u8;
pub type Inc = i8;

// functions

impl Square {

   pub const Size: Sq = 64;

   pub const File_Size: Inc = 8;
   pub const Rank_Size: Inc = 8;

   pub const Stride: Inc = 8;

   pub const None: Self = Self(0); // HACK

   pub const fn new(fl: Inc, rk: Inc) -> Result<Self, ()>{

      if fl >= 0 && fl < Self::File_Size
      && rk >= 0 && rk < Self::Rank_Size {

         Ok(Self((fl * Self::Stride + rk) as Sq))

      } else {

         Err(())
      }
   }

   pub const fn from_int(i: Sq) -> Self {
      debug_assert!(i < Self::Size);
      Self(i)
   }

   pub const fn val(self) -> Sq {
      self.0
   }

   pub const fn index(self) -> usize {
      (self.0 & 0o77) as usize
   }

   pub const fn coords(self) -> (Inc, Inc) {

      let fl = self.0 / Self::Stride as Sq;
      let rk = self.0 % Self::Stride as Sq;

      (fl as Inc, rk as Inc)
   }

   pub const fn file(self) -> Inc {
      self.coords().0
   }

   pub const fn rank(self) -> Inc {
      self.coords().1
   }

   pub const fn sym(self, sd: Side) -> Self {

      if !sd.is_canon() {
         self.rank_opp()
      } else {
         self
      }
   }

   pub const fn is_canon(self) -> bool {
      self.0 < Self::Size / 2
   }

   pub const fn mask_file(self) -> Sq {

      if !self.is_canon() {
         0o70
      } else {
         0o00
      }
   }

   pub const fn mask_rank(sd: Side) -> Sq {

      if !sd.is_canon() {
         0o07
      } else {
         0o00
      }
   }

   pub const fn rank_opp(self) -> Self {
      Self(self.0 ^ 0o07)
   }

   pub const fn ep_opp(self) -> Self {
      Self(self.0 ^ 0o01)
   }

   pub const fn transform(self, mask: Sq) -> Self {
      debug_assert!(mask < Self::Size);
      Self(self.0 ^ mask)
   }

   pub const fn rank_side(self, sd: Side) -> Inc {
      self.sym(sd).rank()
   }

   pub const fn colour(self) -> Colour {
      let (fl, rk) = self.coords();
      Colour::from_int(((fl + rk) & 1) as u8)
   }

   pub const fn is_prom(self) -> bool {
      self.rank() == 0 ||
      self.rank() == Self::Rank_Size - 1
   }

   pub const fn is_prom_side(self, sd: Side) -> bool {
      self.rank_side(sd.opp()) == 0
   }

   pub const fn is_seventh(self, sd: Side) -> bool {
      self.rank_side(sd.opp()) == 1
   }

   pub fn iter() -> impl Iterator<Item = Self> {
      (0 .. Self::Size).map(Self)
   }
}

impl ops::Add<Inc> for Square {

   type Output = Self;

   fn add(self, rhs: Inc) -> Self::Output {
      Self::from_int(self.0.wrapping_add(rhs as Sq))
   }
}

impl ops::Add<(Inc, Inc)> for Square {

   type Output = Option<Self>;

   fn add(self, (df, dr): (Inc, Inc)) -> Self::Output {
      let (fl, rk) = self.coords();
      Self::new(fl + df, rk + dr).ok()
   }
}

impl ops::Sub for Square {

   type Output = Inc;

   fn sub(self, rhs: Self) -> Self::Output {
      self.0.wrapping_sub(rhs.0) as Inc
   }
}

impl ops::Sub<Inc> for Square {

   type Output = Self;

   fn sub(self, rhs: Inc) -> Self::Output {
      Self::from_int(self.0.wrapping_sub(rhs as Sq))
   }
}

impl fmt::Display for Square {

   fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
      let (fl, rk) = self.coords();
      write!(f, "{}{}", util::char_add('a', fl.into()),
                        util::char_add('1', rk.into()))
   }
}

impl str::FromStr for Square {

   type Err = ();

   fn from_str(s: &str) -> Result<Self, ()> {

      if let &[fl, rk] = s.chars()
                          .collect::<Vec<_>>()
                          .as_slice() {

         Self::new(util::char_sub(fl, 'a') as Inc,
                   util::char_sub(rk, '1') as Inc)

      } else {

         Err(())
      }
   }
}

impl Colour {

   pub const Size: u8 = 2;

   pub const fn from_int(i: u8) -> Self {
      debug_assert!(i < Self::Size);
      unsafe { mem::transmute(i) }
   }
}

pub const fn rank_side(rk: Inc, sd: Side) -> Inc {

   debug_assert!(rk >= 0 && rk < Square::Rank_Size);

   if !sd.is_canon() {
      rank_opp(rk)
   } else {
      rk
   }
}

const fn rank_opp(rk: Inc) -> Inc {
   debug_assert!(rk >= 0 && rk < Square::Rank_Size);
   (Square::Rank_Size - 1) - rk
}

pub fn file_from_char(c: char) -> Result<Inc, ()> {

   ('a' ..= 'h').contains(&c)
                .then(|| util::char_sub(c, 'a') as Inc)
                .ok_or(())
}

