
// modules

mod engine;
mod game;
mod prelude;
mod uci;
mod util;

use prelude::*;
use game::{Rule, Table_BB, Table_Hash, Table_Rule};
use engine::{eval::Node, Eval_Def, Table_Eval, Table_Search};

use util::thread::{AtomicF32, Queue};

use std::{io, sync::atomic};

// types

pub struct Global {

   tt_bit: u8,

   input: Queue::<String>,
   seed:  atomic::AtomicU64,

   time_lag:   AtomicF32,
   score_draw: atomic::AtomicI16,

   rule: Rule,

   table_rule:   Table_Rule,
   table_bb:     Table_BB,
   table_hash:   Table_Hash,
   table_eval:   Table_Eval,
   table_search: Table_Search,

   eval: &'static Eval_Def,
}

// functions

fn main() -> io::Result<()> {

   game  ::init();
   engine::init();

   let table_bb   = Table_BB::new();
   let table_eval = Table_Eval::new(&table_bb);

   let global = Box::leak(Box::new(Global {

      tt_bit: 22,

      input: Queue::<String>::new(),
      seed:  1.into(),

      time_lag:   0.1.into(),
      score_draw: 0.into(),

      rule: Rule {
         convert: true.into(),
      },

      table_rule: Table_Rule::new(true),
      table_bb,
      table_hash: Table_Hash::new(),
      table_eval,
      table_search: Table_Search::new(),

      eval: Box::leak(Eval_Def::load_f32("eval")),

   }));

   set_input(&global.input);

   uci::run(global);
   Ok(())
}

impl Global {

   fn next_seed(&self) -> u64 {
      self.seed.fetch_add(1, atomic::Ordering::Relaxed)
   }

   fn attack_bishop(&self, from: Square, blocker: BB) -> BB {
      self.table_bb.attack(Piece::Bishop, from, blocker)
   }

   fn attack_rook(&self, from: Square, blocker: BB) -> BB {
      self.table_bb.attack(Piece::Rook, from, blocker)
   }
}

fn set_input(queue: &'static Queue<String>) {

   std::thread::spawn(move || {

      loop {

         match util::io::read_stdin_line().expect("I/O error") {
            Some(line) => queue.put(line),
            None       => break,
         }
      }

      queue.close();

   });
}

