
// modules

use crate::prelude::*;

use crate::engine::{search};
use crate::game::{Time_Control};

use crate::util::{self, var::Var};

use std::{sync::atomic::Ordering};

// functions

pub fn run(global: &'static Global) {

   let vars = &mut Var::new();

   vars.set("Hash", "64");
   vars.set("Threads", "1");
   vars.set("Move Overhead", "100");
   vars.set("Draw Score", "0");
   vars.set("UCI_Chess960", "false");

   let mut board  = Board::start(global);
   let mut engine = Engine::new(global);

   while let Some(line) = global.input.get() {

      let mut lexer = line.split_whitespace();

      let Some(command) = lexer.next() else {
         error("missing command");
         continue;
      };

      match command {

         "go" => {

            let mut ponder = false;
            let mut wtime = None;
            let mut btime = None;
            let mut winc = 0;
            let mut binc = 0;
            let mut movestogo = None;
            let mut depth = None;
            let mut nodes = None;
            let mut movetime = None;
            let mut infinite = false;

            while let Some(arg) = lexer.next() {

               match arg {
                  "ponder"    => ponder    = true,
                  "wtime"     => wtime     = Some(lexer.next().unwrap().parse::<u32>().unwrap()),
                  "btime"     => btime     = Some(lexer.next().unwrap().parse::<u32>().unwrap()),
                  "winc"      => winc      = lexer.next().unwrap().parse::<u32>().unwrap(),
                  "binc"      => binc      = lexer.next().unwrap().parse::<u32>().unwrap(),
                  "movestogo" => movestogo = Some(lexer.next().unwrap().parse::<u16>().unwrap()),
                  "depth"     => depth     = Some(lexer.next().unwrap().parse::<u8> ().unwrap()),
                  "nodes"     => nodes     = Some(lexer.next().unwrap().parse::<u64>().unwrap()),
                  "movetime"  => movetime  = Some(lexer.next().unwrap().parse::<u32>().unwrap()),
                  "infinite"  => infinite  = true,
                  arg         => error(&format!("unknown argument: {}", arg)),
               }
            }

            assert!(!ponder); // not yet implemented

            let mut si = search::Input {

               thread: vars.get("Threads").parse::<u8>().unwrap(),

               depth: depth.map(Ply),
               node:  nodes,
               time:  movetime.map(|t| t as Time / 1E3),

               score: infinite,

               .. Default::default()
            };

            let (time, inc) = match board.turn() {
               Side::White => (wtime, winc),
               Side::Black => (btime, binc),
            };

            if let Some(time) = time { // time control

               let time = time as Time / 1E3;
               let inc  = inc  as Time / 1E3;

               si.tc = Some(Time_Control {
                  moves: movestogo,
                  time:  time - inc, // convert to pre-increment
                  inc,
               });
            }

            let chess_960 = vars.get("UCI_Chess960").parse::<bool>().unwrap();
            global.rule.convert.store(!chess_960, Ordering::Relaxed);

            let so = engine.search(&board, si);

            if let Some(mv) = so.best.move_ {
               println!("bestmove {}", mv.to_uci(&board));
            } else {
               println!("bestmove {}", "0000");
            }
         }

         "isready" => {

            let tt_size = vars.get("Hash").parse::<u32>().unwrap() << (20 - 4); // * 1MiB / 16 bytes
            engine.resize_tt(util::bit::last(tt_size.into()));

            let time_lag = vars.get("Move Overhead").parse::<u32>().unwrap() as f32 / 1E3;
            global.time_lag.store(time_lag);

            let score_draw = vars.get("Draw Score").parse::<i16>().unwrap();
            global.score_draw.store(score_draw, Ordering::Relaxed);

            println!("readyok");
         }

         "ponderhit" => (),

         "position" => {

            #[derive(Clone, Copy, PartialEq, Eq)]
            enum State { Start, FEN, Moves }

            let mut pos   = String::new();
            let mut moves = String::new();

            let mut state = State::Start;

            while let Some(arg) = lexer.next() {

               match (state, arg) {
                  (_,            "fen")      => state = State::FEN,
                  (_,            "moves")    => state = State::Moves,
                  (State::Start, "startpos") => (),
                  (State::FEN,   arg)        => util::append(&mut pos,   arg),
                  (State::Moves, arg)        => util::append(&mut moves, arg),
                  _ => error(&format!("unknown 'position' argument: {}", arg)),
               }
            }

            board = Board::start(global);

            if !pos.is_empty() {

               match Board::from_fen(&pos, global) {
                  Ok(p)  => board = p,
                  Err(_) => error(&format!("malformed position: {}", pos)),
               }
            }

            for move_ in moves.split_whitespace() {

               match Move::parse(move_, &board) {

                  Ok(mv) => if mv.is_legal(&board) {
                     board.do_move(mv);
                     board.compact();
                  } else {
                     error(&format!("illegal move: {}", move_));
                  }

                  Err(_) => error(&format!("malformed move: {}", move_)),
               }
            }
         }

         "quit" => break,

         "setoption" => {

            #[derive(Clone, Copy, PartialEq, Eq)]
            enum State { Start, Name, Value }

            let mut name  = String::new();
            let mut value = String::new();

            let mut state = State::Start;

            while let Some(arg) = lexer.next() {

               match (state, arg) {
                  (_,            "name")  => state = State::Name,
                  (_,            "value") => state = State::Value,
                  (State::Name,  arg)     => util::append(&mut name,  arg),
                  (State::Value, arg)     => util::append(&mut value, arg),
                  _                       => error(&format!("unknown argument: {}", arg)),
               }
            }

            if name.is_empty() {
               error("missing name");
            } else {
               vars.set(&name, &value);
            }
         }

         "stop" => (),

         "uci" => {

            println!("id name {}", "Senpai 3.0.1");
            println!("id author {}", "Fabien Letouzey");

            println!("option name {} type {} default {} min {} max {}", "Hash", "spin", 1 << 6, 1 << 4, 1 << 15);
            println!("option name {} type {} default {} min {} max {}", "Threads", "spin", 1, 1, 64);
            println!("option name {} type {} default {} min {} max {}", "Move Overhead", "spin", 100, 0, 10_000);
            println!("option name {} type {} default {} min {} max {}", "Draw Score", "spin", 0, -100, 100);
            println!("option name {} type {} default {}", "UCI_Chess960", "check", "false");

            println!("uciok");
         }

         "ucinewgame" => {
            engine.new_game();
            board = Board::start(global);
         }

         _ => error(&format!("unknown command: {}", command)),
      }
   }
}

fn error(message: &str) {
   eprintln!("UCI: {}", message);
}

