
// modules

use super::math;

// constants

const Phi: u64 = 0x9E37_79B9_7F4A_7C15; // ((sqrt(5) - 1) / 2) * 2^64

// types

pub struct Random_Phi(u64);
pub struct Random_Xor(u64);

// functions

pub fn mix(n: impl Into<u64>) -> u64 {
   n.into().wrapping_mul(Phi)
}

pub trait Random : Send {

   fn next_u64(&mut self) -> u64;

   fn next_bits(&mut self, bits: u8) -> u64 {

      debug_assert!(bits <= 64);

      let bits = self.next_u64() >> (64 - bits);
      debug_assert!(bits < (1 << bits));
      bits
   }

   fn next_bool(&mut self, prob: f32) -> bool {

      debug_assert!((0.0 ..= 1.0).contains(&prob));

      let real = self.next_real();
      real < prob
   }

   fn next_int(&mut self, size: u64) -> u64 {
      mod_int(self.next_u64(), size)
   }

   fn next_real(&mut self) -> f32 {

      let real = self.next_u64() as f64 / f64::powi(2.0, 64);
      debug_assert!((0.0 ..= 1.0).contains(&real));
      real as f32
   }

   fn interval(&mut self, min: u64, max: u64) -> u64 {
      debug_assert!(min <= max);
      min + self.next_int(max - min + 1)
   }

   fn uniform(&mut self, min: f32, max: f32) -> f32 {

      debug_assert!(min < max);

      let real = self.next_real();
      math::lerp(min, max, real)
   }

   fn normal(&mut self) -> f32 { // approximation

      let mut sum = 0.0;

      for _ in 0 .. 12 {
         sum += self.next_real();
      }

      sum - 6.0
   }
}

impl Random_Phi { // HACK: evenly distributed but definitely not random #

   pub fn new() -> Self {
      Self(0)
   }
}

impl Default for Random_Phi {

   fn default() -> Self {
      Self::new()
   }
}

impl Random for Random_Phi {

   fn next_u64(&mut self) -> u64 {
      self.0 = self.0.wrapping_add(Phi);
      self.0
   }
}

impl Random_Xor {

   pub fn new() -> Self {
      Self(Phi)
   }

   pub fn from_seed(seed: u64) -> Self {
      Self(mix(seed))
   }
}

impl Default for Random_Xor {

   fn default() -> Self {
      Self::new()
   }
}

impl Random for Random_Xor {

   fn next_u64(&mut self) -> u64 {

      self.0 ^= self.0 >> 12;
      self.0 ^= self.0 << 25;
      self.0 ^= self.0 >> 27;

      self.0.wrapping_mul(0x2545_F491_4F6C_DD1D)
   }
}

pub fn mod_int(mix: u64, size: u64) -> u64 {

   debug_assert!(size != 0);

   let res = ((mix as u128 * size as u128) >> 64) as u64;
   debug_assert!(res < size);
   res
}

